import { ContextService } from '../context';
import { ValueChangeCallback } from '../../models/context/value-change-callback';
import { DeriveContextServiceContract } from '../../models/context/update-context-method';
import { AuthenticatedUser, RestrictedPages, SecurityConfig, OpenIdConfig } from '../../models/security';
import { LifecycleHooks } from '../../providers/service/lifecycle-hooks';
type SecurityContextFields = {
    readonly RESTRICTED_PAGES: string;
    readonly SECURITY_CONFIG: string;
    readonly AUTHENTICATED_USER: string;
    readonly AUTH_TOKEN: string;
    readonly OPEN_ID_CONFIG: string;
};
type SecurityContextFieldParams = {
    readonly RESTRICTED_PAGES: RestrictedPages;
    readonly SECURITY_CONFIG: SecurityConfig;
    readonly AUTHENTICATED_USER: AuthenticatedUser;
    readonly AUTH_TOKEN: string;
    readonly OPEN_ID_CONFIG: OpenIdConfig;
};
/**
 * The SecurityContextService class manages the various fields in the security context.
 */
export declare class SecurityContextService extends ContextService<SecurityContextFields> implements DeriveContextServiceContract<SecurityContextFields, SecurityContextFieldParams>, LifecycleHooks {
    readonly RESTRICTED_PAGES = "restrictedPages";
    readonly SECURITY_CONFIG = "securityConfig";
    readonly AUTHENTICATED_USER = "authenticatedUser";
    readonly AUTH_TOKEN = "jwt";
    readonly OPEN_ID_CONFIG = "openIdConfig";
    /**
     * Retrieves the restricted pages for the user.
     *
     * @return a map with restricted pages.
     */
    getRestrictedPages(): RestrictedPages | undefined;
    /**
     * Updates the restricted pages and notifies subscribers about the change.
     *
     * @param restrictedPages - an object with restricted pages.
     */
    updateRestrictedPages(restrictedPages: RestrictedPages): void;
    /**
     * Registers the <code>callbackFunction</code> to be called whenever the restricted pages are changed.
     *
     * @param callbackFunction - The function to call when the restricted pages are changed.
     * @returns A function to unsubscribe from updates.
     */
    onRestrictedPagesChanged(callbackFunction: ValueChangeCallback<RestrictedPages | undefined>): () => void;
    /**
     * Subscribes to changes in the authentication token.
     *
     * @param callbackFunction - A function to be called when the auth token changes.
     * @returns A function to unsubscribe from updates.
     */
    onAuthTokenChanged(callbackFunction: ValueChangeCallback<string | undefined>): () => void;
    /**
     * Updates the authentication token in the context.
     *
     * @param value - The new auth token to store.
     */
    updateAuthToken(value: string): void;
    /**
     * Retrieves the authentication token from the context.
     *
     * @returns The auth token if available, otherwise undefined.
     */
    getAuthToken(): string | undefined;
    /**
     * Updates the security configuration in the context.
     * @param securityConfig - The new security configuration to be set.
     */
    updateSecurityConfig(securityConfig: SecurityConfig): void;
    /**
     * Subscribes to changes in the security configuration.
     * @param callbackFunction - A function to be called when the security configuration changes.
     * @returns A function that, when called, unsubscribes from the security configuration changes.
     */
    onSecurityConfigChanged(callbackFunction: ValueChangeCallback<SecurityConfig | undefined>): () => void;
    getSecurityConfig(): SecurityConfig | undefined;
    /**
     * Updates the authenticated user information in the context.
     * @param authenticatedUser - The new authenticated user information to be set.
     */
    updateAuthenticatedUser(authenticatedUser: AuthenticatedUser): void;
    /**
     * Subscribes to changes in the authenticated user information.
     * @param callbackFunction - A function to be called when the authenticated user information changes.
     * @returns A function that, when called, unsubscribes from the authenticated user information changes.
     */
    onAuthenticatedUserChanged(callbackFunction: ValueChangeCallback<AuthenticatedUser | undefined>): () => void;
    /**
     * Retrieves the authenticated user information.
     * @return the authenticated user information or undefined, if there is no user.
     */
    getAuthenticatedUser(): AuthenticatedUser | undefined;
    /**
     * Updates the OpenID configuration in the context.
     *
     * @param openIdConfig - The new OpenID configuration to set.
     */
    updateOpenIdConfig(openIdConfig: OpenIdConfig): void;
    /**
     * Retrieves the OpenID configuration from the context.
     *
     * @returns The current OpenID configuration or undefined.
     */
    getOpenIdConfig(): OpenIdConfig | undefined;
}
export {};
