/*
 * Decompiled with CFR 0.152.
 */
package com.ontotext.models.query;

import com.ontotext.models.query.LangFilter;
import com.ontotext.soaas.common.ObjectsUtil;
import java.io.IOException;
import java.io.InputStream;
import java.lang.invoke.MethodHandles;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.util.Collections;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.IllformedLocaleException;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Consumer;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.commons.io.IOUtils;
import org.apache.commons.io.LineIterator;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class LangValidator {
    private static final Logger LOGGER = LoggerFactory.getLogger(MethodHandles.lookup().lookupClass());
    private static final Set<String> FORBIDDEN_LANGS = Stream.of("ALL", "ANY").collect(Collectors.toSet());
    private static final EnumSet<LangFilter.TokenType> ALLOWED_TYPES = EnumSet.of(LangFilter.TokenType.EXACT_MATCH, new LangFilter.TokenType[]{LangFilter.TokenType.START_WITH, LangFilter.TokenType.NO_LANG, LangFilter.TokenType.ANY_LANG, LangFilter.TokenType.ALL, LangFilter.TokenType.UNIQUE});
    private static final IanaValidator IANA_VALIDATOR = new IanaValidator();
    public static final LangValidator ALWAYS_VALID = new LangValidator(null){

        @Override
        public void validate(List<Pair<String, String>> values, Consumer<Pair<String, String>> notAllowed, Consumer<Pair<String, String>> excluded) {
        }
    };
    private final List<LangFilter.Token> tokens;

    LangValidator(List<LangFilter.Token> tokens) {
        this.tokens = (List)ObjectsUtil.getOrDefaultGet(tokens, Collections::emptyList);
        this.checkForInvalidTokens(tokens);
    }

    private void checkForInvalidTokens(List<LangFilter.Token> tokens) {
        if (tokens == null) {
            return;
        }
        String notAllowedTokens = tokens.stream().filter(token -> !ALLOWED_TYPES.contains((Object)token.getType())).map(Objects::toString).collect(Collectors.joining(", "));
        if (!notAllowedTokens.isEmpty()) {
            throw new IllegalArgumentException(String.format("The use of %s is not allow for validation", notAllowedTokens));
        }
        if (tokens.stream().anyMatch(token -> token.getType() == LangFilter.TokenType.UNIQUE && token.isExclusion())) {
            throw new IllegalArgumentException("It's not allowed to check for none duplicates using '-UNIQ'");
        }
        if (tokens.stream().allMatch(LangFilter.Token::isExclusion)) {
            String exclusions = tokens.stream().map(Objects::toString).collect(Collectors.joining(", "));
            throw new IllegalArgumentException(String.format("It's not allowed for the validation pattern to contain only exclusions %s", exclusions));
        }
    }

    public static boolean isValidLanguageTag(String lang) {
        if (lang == null || lang.isEmpty() || FORBIDDEN_LANGS.contains(lang)) {
            return false;
        }
        try {
            Locale locale = new Locale.Builder().setLanguageTag(lang).build();
            return IANA_VALIDATOR.isValid(locale);
        }
        catch (IllformedLocaleException ife) {
            return false;
        }
    }

    public List<LangFilter.Token> getTokens() {
        return this.tokens;
    }

    public LangFilter toFilter() {
        return new LangFilter(this.tokens, LangFilter.FilterType.VALIDATE);
    }

    public void validate(List<Pair<String, String>> values, Consumer<Pair<String, String>> notAllowed, Consumer<Pair<String, String>> excluded) {
        Set<String> langs = values.stream().map(Pair::getValue).map(String::toLowerCase).collect(Collectors.toSet());
        LinkedHashSet notAllowedLangs = new LinkedHashSet(langs);
        LinkedList<LangFilter.Token> tokensCopy = new LinkedList<LangFilter.Token>(this.tokens);
        tokensCopy.removeIf(token -> token.getType() == LangFilter.TokenType.UNIQUE);
        Iterator tokenIterator = tokensCopy.iterator();
        while (tokenIterator.hasNext()) {
            LangFilter.Token token2 = (LangFilter.Token)tokenIterator.next();
            if (!token2.isExclusion()) continue;
            token2.applyExclusion(langs);
            tokenIterator.remove();
        }
        notAllowedLangs.removeAll(langs);
        values.stream().filter(pair -> notAllowedLangs.contains(((String)pair.getRight()).toLowerCase())).forEach(excluded);
        if (tokensCopy.isEmpty()) {
            return;
        }
        Set validLangs = tokensCopy.stream().flatMap(token -> token.matchValue(langs, true)).filter(Objects::nonNull).collect(Collectors.toSet());
        values.stream().filter(value -> {
            String lang = ((String)value.getRight()).toLowerCase();
            return !validLangs.contains(lang) && !notAllowedLangs.contains(lang);
        }).forEach(notAllowed);
    }

    public List<Pair<String, String>> validateUniqueness(List<Pair<String, String>> values) {
        if (this.tokens.stream().noneMatch(token -> token.getType() == LangFilter.TokenType.UNIQUE)) {
            return Collections.emptyList();
        }
        Map<String, Long> groupByLang = values.stream().collect(Collectors.groupingBy(Pair::getRight, Collectors.counting()));
        groupByLang.entrySet().removeIf(entry -> (Long)entry.getValue() == 1L);
        return values.stream().filter(pair -> groupByLang.containsKey(pair.getRight())).collect(Collectors.toList());
    }

    private static class IanaValidator {
        private static final String IANA_SUBTAG_REGISTRY_CACHE_LOCATION = "iana-language-subtag-registry.txt";
        private static final String LANGUAGE_TYPE = "language";
        private static final String SCRIPT_TYPE = "script";
        private static final String REGION_TYPE = "region";
        private static final String VARIANT_TYPE = "variant";
        private static final String ENTRY_SEPARATOR = "%%";
        private static final String KEY_SEPARATOR = ": ";
        private Map<String, IanaEntry> languages = new HashMap<String, IanaEntry>(0, 0.99f);
        private Map<String, IanaEntry> scripts = new HashMap<String, IanaEntry>(0, 0.99f);
        private Map<String, IanaEntry> regions = new HashMap<String, IanaEntry>(0, 0.99f);
        private Map<String, IanaEntry> variants = new HashMap<String, IanaEntry>(0, 0.99f);

        IanaValidator() {
            this.initialize();
        }

        private void initialize() {
            this.load(this::distributeInCache);
        }

        private void load(Consumer<IanaEntry> entryConsumer) {
            String from;
            AtomicInteger count = new AtomicInteger();
            Consumer<IanaEntry> consumer = entryConsumer.andThen(ianaEntry -> count.incrementAndGet());
            try (InputStream source = this.getClass().getClassLoader().getResourceAsStream(IANA_SUBTAG_REGISTRY_CACHE_LOCATION);){
                assert (source != null) : "Could not load the iana-language-subtag-registry.txt";
                LineIterator lineIterator = IOUtils.lineIterator((InputStream)source, (Charset)StandardCharsets.UTF_8);
                from = lineIterator.nextLine();
                IanaEntry entry = null;
                while (lineIterator.hasNext()) {
                    String line = lineIterator.nextLine();
                    entry = IanaValidator.readLine(consumer, line, entry);
                }
            }
            catch (IOException ioe) {
                throw new IllegalStateException(ioe);
            }
            String[] pair = IanaValidator.splitEntryLine(from);
            LOGGER.info("Successfully loaded IANA Subtag Registry cache from {}. Loaded {} entries.", (Object)pair[1], (Object)count);
        }

        private static IanaEntry readLine(Consumer<IanaEntry> entryConsumer, String line, IanaEntry entry) {
            String value;
            String key;
            if (line.equals(ENTRY_SEPARATOR)) {
                if (entry != null) {
                    entryConsumer.accept(entry);
                }
                return new IanaEntry();
            }
            if (entry == null) {
                return null;
            }
            String[] pair = IanaValidator.splitEntryLine(line);
            if (line.startsWith(" ")) {
                key = "Comments";
                value = pair[0];
            } else {
                key = pair[0];
                value = pair[1];
            }
            entry.add(key, value);
            return entry;
        }

        private void distributeInCache(IanaEntry entry) {
            String type;
            switch (type = entry.getType()) {
                case "language": {
                    this.languages.put(entry.getSubtag(), entry);
                    break;
                }
                case "script": {
                    this.scripts.put(entry.getSubtag(), entry);
                    break;
                }
                case "region": {
                    this.regions.put(entry.getSubtag(), entry);
                    break;
                }
                case "variant": {
                    this.variants.put(entry.getSubtag(), entry);
                    break;
                }
            }
        }

        private static String[] splitEntryLine(String from) {
            return from.split(KEY_SEPARATOR, 2);
        }

        public boolean isValid(Locale locale) {
            boolean isValid = this.isValidToken(locale.getLanguage(), this.languages::containsKey);
            if (!isValid) {
                return false;
            }
            isValid = this.isValidToken(locale.getScript(), this.scripts::containsKey);
            if (!isValid) {
                return false;
            }
            isValid = this.isValidToken(locale.getCountry(), this.regions::containsKey);
            if (!isValid) {
                return false;
            }
            return this.isValidVariant(locale.getVariant());
        }

        private boolean isValidVariant(String variant) {
            if (this.isValidToken(variant, this.variants::containsKey)) {
                return true;
            }
            String[] parts = variant.split("_");
            if (parts.length > 1) {
                return Stream.of(parts).allMatch(this.variants::containsKey);
            }
            return false;
        }

        private boolean isValidToken(String token, Predicate<String> tokenMappings) {
            if (StringUtils.isBlank((CharSequence)token)) {
                return true;
            }
            return tokenMappings.test(token);
        }
    }

    private static class IanaEntry
    extends HashMap<String, Object> {
        private static final String TYPE = "Type";
        private static final String TAG = "Tag";
        private static final String SUBTAG = "Subtag";
        private static final String COMMENTS = "Comments";
        private static final String PREFIX = "Prefix";
        private static final String DESCRIPTION = "Description";
        private static final String PREFERRED_VALUE = "Preferred-Value";
        private static final Set<String> MULTI_VALUE_KEYS = Stream.of("Prefix", "Description", "Comments").collect(Collectors.toSet());

        private IanaEntry() {
        }

        String getType() {
            return (String)this.get(TYPE);
        }

        String getTag() {
            return (String)this.get(TAG);
        }

        String getSubtag() {
            return (String)this.get(SUBTAG);
        }

        List<String> getDescription() {
            return (List)this.get(DESCRIPTION);
        }

        List<String> getPrefix() {
            return (List)this.get(PREFIX);
        }

        String getPreferredValue() {
            return (String)this.get(PREFERRED_VALUE);
        }

        List<String> getComments() {
            return (List)this.get(COMMENTS);
        }

        static boolean isMultiValuedKey(String key) {
            return MULTI_VALUE_KEYS.contains(key);
        }

        private void add(String key, String value) {
            if (IanaEntry.isMultiValuedKey(key)) {
                ((List)this.computeIfAbsent(key, k -> new LinkedList())).add(value);
            } else {
                this.put(key, value);
            }
        }
    }
}

