/*
 * Decompiled with CFR 0.152.
 */
package com.ontotext.graphdb.raft.node.task;

import com.ontotext.graphdb.raft.NodeState;
import com.ontotext.graphdb.raft.grpc.RaftRpcConnectionException;
import com.ontotext.graphdb.raft.grpc.RpcNodeClient;
import com.ontotext.graphdb.raft.grpc.VerifyEntry;
import com.ontotext.graphdb.raft.grpc.VerifyResponse;
import com.ontotext.graphdb.raft.node.Quorum;
import com.ontotext.graphdb.raft.node.RaftTaskController;
import com.ontotext.graphdb.raft.node.concurrent.QuorumStateException;
import com.ontotext.graphdb.raft.recovery.RaftRecoveryException;
import com.ontotext.graphdb.raft.storage.LogEntry;
import java.util.EnumMap;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.locks.ReentrantLock;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class VerifyEntryTask
implements Runnable {
    private static final Logger logger = LoggerFactory.getLogger(VerifyEntryTask.class);
    private final RaftTaskController controller;
    private final LogEntry entry;
    private final Quorum quorum;
    private final VerifyEntry verifyEntry;
    private final VerifyResponse currentEntryState;
    private volatile RpcNodeClient leaderClient;
    private volatile Map<String, Integer> fingerprints;
    private volatile Map<Long, Integer> lastIndexes;
    private volatile Map<VerifyResponse.Status, Integer> status;
    private volatile int success;
    private ReentrantLock lock;
    private final boolean failedToCommit;
    private final boolean synchronous;

    public VerifyEntryTask(RaftTaskController controller, LogEntry entry, Quorum quorum, RpcNodeClient leaderClient, boolean failedToCommit, boolean synchronous) {
        this(controller, entry, quorum, VerifyEntry.newBuilder().setLogIndex(entry.getIndex()).build(), leaderClient, null, null, null, null, failedToCommit, synchronous);
    }

    private VerifyEntryTask(RaftTaskController controller, LogEntry entry, Quorum quorum, VerifyEntry verifyEntry, RpcNodeClient leaderClient, Map<String, Integer> quorumFingerprints, Map<Long, Integer> lastIndexes, Map<VerifyResponse.Status, Integer> status, ReentrantLock lock, boolean failedToCommit, boolean synchronous) {
        this.controller = controller;
        this.entry = entry;
        this.quorum = quorum;
        this.verifyEntry = verifyEntry;
        this.leaderClient = leaderClient;
        this.fingerprints = quorumFingerprints;
        this.lastIndexes = lastIndexes;
        this.status = status;
        this.lock = lock;
        this.success = -1;
        this.failedToCommit = failedToCommit;
        this.synchronous = synchronous;
        this.currentEntryState = this.buildCurrentEntryState();
    }

    @Override
    public void run() {
        this.verifyEntry();
    }

    private void verifyEntry() {
        try {
            do {
                if (this.controller.getNodeState() == NodeState.OUT_OF_SYNC || Thread.currentThread().isInterrupted()) {
                    logger.warn("[Node {}] Validation for entry {} aborted. Unlocking channel {}", new Object[]{this.controller.getCurrentAddress(), this.entry.getIndex(), this.entry.getChannel()});
                    this.controller.getTransactionLog().unlockChannel(this.entry.getChannel());
                    return;
                }
                if (this.controller.getTransactionLog().fetchEntryStatus(this.entry.getIndex()) == LogEntry.Status.VALID) {
                    logger.warn("Entry {} is already valid. No need to double-verify it.", (Object)this.entry.getIndex());
                    this.success = 1;
                    continue;
                }
                this.doActualVerification();
            } while (this.synchronous && this.success < 0);
            if (this.success < 0 && !this.synchronous) {
                this.scheduleAsyncVerification();
            } else {
                this.controller.releaseQuorum(this.quorum);
            }
        }
        catch (Exception e) {
            logger.error("Unexpected error occurred during verification of entry {}", (Object)this.entry.getIndex(), (Object)e);
            throw e;
        }
    }

    private void doActualVerification() {
        try {
            if (this.leaderClient != null) {
                if (logger.isDebugEnabled()) {
                    logger.debug("Verifying entry {} status with leader {} ", (Object)this.entry.getIndex(), (Object)this.leaderClient.getAddress());
                }
                this.verifyWithLeader();
            } else {
                if (this.lock == null) {
                    this.lock = new ReentrantLock();
                    this.fingerprints = new HashMap<String, Integer>();
                    this.status = new EnumMap<VerifyResponse.Status, Integer>(VerifyResponse.Status.class);
                    this.lastIndexes = new HashMap<Long, Integer>();
                }
                if (logger.isDebugEnabled()) {
                    logger.debug("Verifying entry {} status with cluster quorum and current status {} ", (Object)this.entry.getIndex(), (Object)this.controller.getNodeState());
                }
                this.verifyWithQuorum();
            }
            if (this.success < 0 && this.synchronous) {
                try {
                    Thread.sleep(this.controller.getVerificationTimeout());
                }
                catch (InterruptedException e) {
                    Thread.currentThread().interrupt();
                    logger.error("Attempt to interrupt synchronous verification for log {} ", (Object)this.entry.getIndex());
                }
            }
        }
        catch (RaftRecoveryException e) {
            logger.error("Unable to verify entry due to recovery phase exception", (Throwable)((Object)e));
            throw e;
        }
        catch (Exception e) {
            logger.error("Unable to verify entry. Trying again", (Throwable)e);
        }
        finally {
            if (this.success == 0) {
                this.controller.getTransactionLog().unlockChannel(this.entry.getChannel());
            }
        }
    }

    private void scheduleAsyncVerification() {
        this.controller.scheduleTaskInMs(new VerifyEntryTask(this.controller, this.entry, this.quorum, this.verifyEntry, this.leaderClient, this.fingerprints, this.lastIndexes, this.status, this.lock, this.failedToCommit, this.synchronous), this.controller.getVerificationTimeout());
    }

    private void verifyWithLeader() {
        try {
            VerifyResponse response = this.leaderClient.sendValidateRpc(this.verifyEntry);
            if (response.getStatus() == VerifyResponse.Status.VALID) {
                if (!this.failedToCommit && this.controller.areFingerprintsEqual(this.entry.getFingerprint(), response.getFingerprint())) {
                    if (this.entry.getChannel() == -2) {
                        if (this.entry.isMembershipConfigEntry()) {
                            this.controller.updateClusterGroup(this.entry.getOldConfig(), this.entry.getNewConfig());
                        } else {
                            this.controller.updateClusterGroup(this.entry.getConfigParameters());
                        }
                    }
                    this.controller.getTransactionLog().validateEntry(this.entry);
                    this.success = 1;
                    logger.info("Successfully verified entry {} with leader node", (Object)this.entry.getIndex());
                } else {
                    if (this.failedToCommit) {
                        logger.error("Unsuccessful verification of entry {} with leader as node failed to commit transaction", (Object)this.entry.getIndex());
                    } else {
                        logger.error("Unsuccessful verification of entry {} with leader due to mismatch in fingerprint where expected: {} but actual: {}", new Object[]{this.entry.getIndex(), response.getFingerprint(), this.entry.getFingerprint()});
                    }
                    this.goOutOfSync();
                }
            } else if (response.getStatus() != VerifyResponse.Status.PROCESSED) {
                logger.error("Entry {} has been replaced by another with status {}", (Object)this.entry.getIndex(), (Object)response.getStatus());
                this.goOutOfSync();
            }
        }
        catch (RaftRpcConnectionException e) {
            this.leaderClient.setNoConnectionStatus(e);
            logger.error("Could not verify entry {} with leader {} due to: {}", new Object[]{this.entry.getIndex(), this.leaderClient.getAddress(), e.getMessage()});
            this.leaderClient = null;
        }
    }

    private void verifyWithQuorum() {
        this.prepareForNewVerification();
        logger.info("[Node {}] Verifying entry {} with quorum from other nodes and status {}", new Object[]{this.controller.getCurrentAddress(), this.entry.getIndex(), this.controller.getNodeState()});
        this.processQuorumVerifyResponse(this.currentEntryState, null);
        for (RpcNodeClient rpcNode : this.controller.getRpcNodes()) {
            if (this.isConfigUpdateWithTotalConsistency() && this.entry.isMembershipConfigEntry()) {
                if (!this.isNodePartOfNewConfig(rpcNode)) continue;
                this.quorum.addTask(this.controller.submitTask(() -> this.verifyEntryWithNode(rpcNode)));
                continue;
            }
            this.quorum.addTask(this.controller.submitTask(() -> this.verifyEntryWithNode(rpcNode)));
        }
        if (this.quorum.getTotal()) {
            this.quorum.awaitAll();
        } else {
            this.quorum.await();
        }
        if (this.quorum.getState() == Quorum.State.SUCCESSFUL) {
            if (!this.failedToCommit) {
                if (this.entry.getChannel() == -2 && !this.entry.isMembershipConfigEntry()) {
                    this.controller.updateClusterGroup(this.entry.getConfigParameters());
                }
                this.controller.getTransactionLog().validateEntry(this.entry);
                this.success = 1;
                logger.info("Successfully verified entry {} with quorum", (Object)this.entry.getIndex());
            } else {
                logger.error("Unsuccessful verification of entry {} with quorum as node failed to commit entry", (Object)this.entry.getIndex());
                this.goOutOfSync();
            }
        } else if (this.quorum.getState() == Quorum.State.UNSUCCESSFUL) {
            logger.error("Unsuccessful verification of entry {} with quorum", (Object)this.entry.getIndex());
            this.goOutOfSync();
        }
    }

    private void prepareForNewVerification() {
        this.fingerprints.clear();
        this.lastIndexes.clear();
        this.status.clear();
        this.quorum.clear();
    }

    private boolean isConfigUpdateWithTotalConsistency() {
        return this.quorum.getTotal() && this.entry.getChannel() == -2;
    }

    private void verifyEntryWithNode(RpcNodeClient rpcNode) {
        try {
            if (logger.isDebugEnabled()) {
                logger.debug("Sending entry verification to {} with term {}", (Object)rpcNode.getAddress(), (Object)this.controller.getCurrentTerm());
            }
            VerifyResponse response = rpcNode.sendValidateRpc(this.verifyEntry);
            if (rpcNode.getStatus() == RpcNodeClient.Status.NO_CONNECTION) {
                rpcNode.setInSyncStatus();
            }
            this.processQuorumVerifyResponse(response, rpcNode);
        }
        catch (QuorumStateException e) {
            logger.error("Cannot send message to {} due to: {}", (Object)rpcNode.getAddress(), (Object)e.getMessage());
        }
        catch (Exception e) {
            rpcNode.setNoConnectionStatus(e);
            logger.error("Cannot send message to {} due to: {}", (Object)rpcNode.getAddress(), (Object)e.getMessage());
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void processQuorumVerifyResponse(VerifyResponse response, RpcNodeClient client) {
        if (response.getStatus() == VerifyResponse.Status.VALID || response.getStatus() == VerifyResponse.Status.PROCESSED) {
            this.lock.lock();
            try {
                Integer count = this.fingerprints.putIfAbsent(response.getFingerprint(), 1);
                if (count != null) {
                    this.fingerprints.put(response.getFingerprint(), count + 1);
                }
            }
            finally {
                this.lock.unlock();
            }
            this.quorum.increment(this.isEntryMatching(response));
        } else if (response.getStatus() != VerifyResponse.Status.PROCESSING && !this.isEntryCreated(response)) {
            if (response.getStatus() == VerifyResponse.Status.NOT_REACHED) {
                this.lock.lock();
                try {
                    Integer count = this.lastIndexes.putIfAbsent(response.getLogIndex(), 1);
                    if (count != null) {
                        this.lastIndexes.put(response.getLogIndex(), count + 1);
                    }
                    if ((count = this.status.putIfAbsent(response.getStatus(), 1)) != null) {
                        this.status.put(response.getStatus(), count + 1);
                    }
                }
                finally {
                    this.lock.unlock();
                }
            }
            if (!this.isConfigUpdateWithTotalConsistency() && (this.controller.getNodeState() != NodeState.LEADER || client != null && client.getStatus() == RpcNodeClient.Status.OUT_OF_SYNC)) {
                this.quorum.increment(false);
            }
        }
    }

    private void goOutOfSync() {
        if (this.entry.getChannel() == -2 && this.entry.isMembershipConfigEntry()) {
            this.controller.rollbackClusterGroupUpdate(this.entry.getOldConfig(), this.entry.getNewConfig());
        }
        this.controller.goOutOfSync();
    }

    private boolean isEntryCreated(VerifyResponse response) {
        return response.getStatus() == VerifyResponse.Status.CREATED && this.isEntryMatching(response);
    }

    private boolean isEntryMatching(VerifyResponse response) {
        return this.controller.areFingerprintsEqual(this.entry.getFingerprint(), response.getFingerprint()) && response.getLogIndex() == this.verifyEntry.getLogIndex() && response.getChannel() == this.entry.getChannel() && response.getLogTerm() == this.entry.getTerm();
    }

    private boolean isNodePartOfNewConfig(RpcNodeClient rpcNode) {
        return this.entry.getNewConfig().stream().anyMatch(node -> node.getRpcAddress().equals(rpcNode.getAddress()));
    }

    private VerifyResponse buildCurrentEntryState() {
        VerifyResponse.Builder builder = VerifyResponse.newBuilder();
        builder.setLogIndex(this.entry.getIndex());
        builder.setFingerprint(this.entry.getFingerprint());
        builder.setChannel(this.entry.getChannel());
        builder.setLogTerm(this.entry.getTerm());
        builder.setStatus(VerifyResponse.Status.valueOf(this.entry.getStatus().name()));
        return builder.build();
    }
}

