import { Service } from '../../providers/service/service';
import { Authority } from '../../models/security';
import { Repository } from '../../models/repositories';
/**
 * Service responsible for handling authentication-related operations.
 */
export declare class AuthenticationService implements Service {
    private readonly repositoryService;
    private readonly securityContextService;
    login(): string;
    /**
     * Updates security context for logout request.
     */
    logout(): void;
    /**
     * Checks if the user is logged in based on the provided configuration and user details.
     * @returns {boolean} True if the user is authenticated, false otherwise.
     */
    isLoggedIn(): boolean;
    /**
     * Check if the user is authenticated.
     * A user is considered authenticated, if security is disabled, if he is external, or if there is an
     * auth token in the store
     */
    isAuthenticated(): boolean;
    /**
     * Determines if free access is allowed based on the security configuration.
     * @returns {boolean | undefined} True if free access is enabled, false or undefined otherwise.
     */
    hasFreeAccess(): boolean | undefined;
    /**
     * Checks if the user has a specific role based on the provided authority, configuration, and user details.
     * @param {Authority} role - The authority role to check.
     * @returns {boolean | undefined} True if the user has the specified role, false or undefined otherwise.
     */
    hasRole(role?: Authority): boolean | undefined;
    /**
     * Checks if security is enabled
     * @returns {boolean} True if security is enabled, false otherwise.
     */
    isSecurityEnabled(): boolean;
    /**
     * Checks if the current user has read permissions for the specified repository.
     * This method evaluates if the user can read the repository based on security configuration,
     * user authentication status, and user roles.
     *
     * @param {Repository} repository - The repository to check read permissions for.
     * @returns {boolean} True if the user has read permissions for the repository, false otherwise.
     */
    canReadRepo(repository?: Repository): boolean;
    /**
     * Checks if the current user has write permissions for the specified repository.
     * This method evaluates if the user can write to the repository based on security configuration,
     * user authentication status, and user roles.
     *
     * @param {Repository} repository - The repository to check write permissions for. Optional parameter.
     * @returns {boolean} True if the user has write permissions for the repository, false otherwise.
     */
    canWriteRepo(repository?: Repository): boolean;
    /**
     * Checks if the current user has GraphQL read permissions for the specified repository.
     * This method determines if the user can execute GraphQL read operations on the repository.
     *
     * @param {Repository} repository - The repository to check GraphQL read permissions for.
     * @returns {boolean} True if the user has GraphQL read permissions for the repository, false otherwise.
     */
    canReadGqlRepo(repository: Repository): boolean;
    /**
     * Checks if the current user has GraphQL write permissions for the specified repository.
     * This method determines if the user can execute GraphQL write operations on the repository.
     *
     * @param {Repository} repository - The repository to check GraphQL write permissions for.
     * @returns {boolean} True if the user has GraphQL write permissions for the repository, false otherwise.
     */
    canWriteGqlRepo(repository: Repository): boolean;
    /**
     * Checks if the current user has any GraphQL permissions (read or write) for the specified repository.
     * This is a convenience method that combines the results of canReadGqlRepo and canWriteGqlRepo.
     *
     * @param {Repository} repository - The repository to check GraphQL permissions for.
     * @returns {boolean} True if the user has any GraphQL permissions for the repository, false otherwise.
     */
    hasGqlRights(repository: Repository): boolean;
    /**
     * Determines if the current user has authority to access the active route.
     *
     * This method checks if the user has the necessary permissions to access the current route
     * based on the route's defined authority requirements and the user's assigned roles.
     * The method follows these rules:
     * - If no active route exists, access is denied
     * - Admin users always have access to all routes
     * - Routes without defined authority requirements are accessible to all
     * - If no repository is selected, authority checks are bypassed
     * - If the user has any of the authorities required by the route, access is granted
     *
     * @returns {boolean} True if the user has authority to access the current route, false otherwise
     */
    hasAuthority(): boolean;
    private resolveAuthorities;
    private hasGraphqlAuthority;
    private isAdminOrRepoManager;
    private getSecurityConfig;
    private getAuthenticatedUser;
    private hasBaseRights;
    private checkRepoRights;
}
