/*
 * Decompiled with CFR 0.152.
 */
package com.ontotext.graphdb.raft.util;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.annotations.VisibleForTesting;
import com.ontotext.graphdb.Config;
import com.ontotext.graphdb.raft.ClusterGroup;
import com.ontotext.graphdb.raft.NodeState;
import com.ontotext.graphdb.raft.RaftException;
import com.ontotext.graphdb.raft.config.ConfigUtil;
import com.ontotext.graphdb.raft.grpc.AppendEntry;
import com.ontotext.graphdb.raft.grpc.AppendResponse;
import com.ontotext.graphdb.raft.grpc.ClusterConfigOptions;
import com.ontotext.graphdb.raft.grpc.NodeInfo;
import com.ontotext.graphdb.raft.grpc.RaftRpcConnectionException;
import com.ontotext.graphdb.raft.grpc.RpcNodeClient;
import com.ontotext.graphdb.raft.node.Quorum;
import com.ontotext.graphdb.raft.node.RaftTaskController;
import com.ontotext.graphdb.raft.node.task.CatchupTask;
import com.ontotext.graphdb.raft.recovery.RaftRecoveryException;
import jakarta.annotation.Nullable;
import java.io.File;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.charset.StandardCharsets;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.Future;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class RaftUtil {
    private static final Logger logger = LoggerFactory.getLogger(RaftUtil.class);
    private static volatile int NODE_SYNC_TIMEOUT = Config.getPropertyAsInt((String)"graphdb.raft.node.sync.timeout", (int)5000);
    private static volatile int SYSTEM_HEARTBEAT_TIMEOUT = Config.getPropertyAsInt((String)"graphdb.raft.system.heartbeat.timeout", (int)30000);

    private RaftUtil() {
    }

    public static boolean isHeartbeat(AppendEntry entry) {
        return entry.getChannel() == -1 || entry.getChannel() == -7;
    }

    public static byte[] serializeString(String value) {
        byte[] bytes = value.getBytes(StandardCharsets.UTF_8);
        ByteBuffer buff = ByteBuffer.allocate(bytes.length + 4);
        buff.putInt(bytes.length);
        buff.put(bytes);
        return buff.array();
    }

    public static void validateEnoughDiskSpace(long expectedSize, File snapshotLocation) {
        if ((double)snapshotLocation.getUsableSpace() <= (double)expectedSize * 1.5) {
            throw new RaftRecoveryException("Node doesn't have enough disk space to replicate snapshot. Expected at least " + (double)expectedSize * 1.5 + " but only " + snapshotLocation.getUsableSpace() + " was available.");
        }
    }

    @VisibleForTesting
    public static void setSystemSyncTimeout(int syncTimeout) {
        SYSTEM_HEARTBEAT_TIMEOUT = syncTimeout;
    }

    @VisibleForTesting
    public static void setNodeSyncTimeout(int syncTimeout) {
        NODE_SYNC_TIMEOUT = syncTimeout;
    }

    public static boolean isSystemHeartbeat(AppendEntry entry) {
        return entry.getChannel() == -7;
    }

    public static RpcNodeClient getPrimaryLeader(RaftTaskController controller) {
        ClusterGroup group = controller.getGroup();
        RpcNodeClient primary = null;
        Collection<RpcNodeClient> primaries = group.getPrimaryNodes();
        for (RpcNodeClient client : primaries) {
            try {
                String leader = client.getLeader();
                if (leader.isBlank()) continue;
                primary = group.getPrimaryRpcNode(leader);
            }
            catch (RaftRpcConnectionException e) {
                logger.warn("Primary cluster node {} is not available", (Object)client.getAddress());
            }
        }
        if (primary == null) {
            throw new RaftException("Primary leader not found");
        }
        return primary;
    }

    public static void handleOutOfSyncFollower(RpcNodeClient rpcNode, long matchIndex, Quorum quorum, RaftTaskController state, boolean awaitToSync) {
        if (state.getNodeState() != NodeState.LEADER) {
            if (rpcNode.getStatus() == RpcNodeClient.Status.OUT_OF_SYNC) {
                rpcNode.setInSyncStatus();
            }
            logger.warn("[Node {}] Cannot handle out of sync followers as node is no longer leader", (Object)state.getCurrentAddress());
            return;
        }
        long lastLogIndex = state.getLastLogIndex();
        if (matchIndex < lastLogIndex) {
            if (logger.isDebugEnabled()) {
                logger.debug("Catching up node {} from index {} to index {}", new Object[]{rpcNode.getAddress(), matchIndex, lastLogIndex});
            }
            if (awaitToSync) {
                RaftUtil.awaitNodesToSync(rpcNode.getAddress());
            }
            RaftUtil.catchUpNode(quorum, rpcNode, matchIndex, state);
        } else {
            if (quorum != null) {
                quorum.increment(true);
            }
            rpcNode.setInSyncStatus();
        }
    }

    public static boolean isClusterInSync(Quorum quorum, RaftTaskController controller) {
        try {
            int count = 0;
            quorum.setTotal(true);
            AppendEntry heartbeat = RaftUtil.buildSystemHeartbeat(controller);
            while (quorum.getState() != Quorum.State.SUCCESSFUL) {
                RaftUtil.sendSystemHeartbeat(heartbeat, quorum, controller);
                if (quorum.getState() == Quorum.State.SUCCESSFUL) continue;
                if (count > 6) {
                    return false;
                }
                RaftUtil.awaitNodesToSync(null);
                ++count;
                quorum.clear();
            }
            quorum.clear();
            quorum.setTotal(false);
            return true;
        }
        catch (Exception e) {
            String msg = e.getMessage() != null ? e.getMessage() : "";
            logger.warn("Error occurred during system heartbeat synchronization: {}", (Object)msg);
            return false;
        }
    }

    public static void updateClusterGroup(List<NodeInfo> oldConfig, List<NodeInfo> newConfig, ClusterGroup clusterGroup) {
        Set<NodeInfo> added = ConfigUtil.getDifference(new HashSet<NodeInfo>(newConfig), new HashSet<NodeInfo>(oldConfig));
        Set<NodeInfo> removed = ConfigUtil.getDifference(new HashSet<NodeInfo>(oldConfig), new HashSet<NodeInfo>(newConfig));
        added.forEach(clusterGroup::addClusterRpcNode);
        removed.forEach(clusterGroup::removeClusterRpcNode);
    }

    public static void rollbackClusterGroupUpdate(List<NodeInfo> oldConfig, List<NodeInfo> newConfig, ClusterGroup clusterGroup) {
        Set<NodeInfo> added = ConfigUtil.getDifference(new HashSet<NodeInfo>(newConfig), new HashSet<NodeInfo>(oldConfig));
        Set<NodeInfo> removed = ConfigUtil.getDifference(new HashSet<NodeInfo>(oldConfig), new HashSet<NodeInfo>(newConfig));
        added.forEach(clusterGroup::removeClusterRpcNode);
        removed.forEach(clusterGroup::addClusterRpcNode);
    }

    public static ClusterConfigOptions buildClusterConfigOptions(Map<String, Number> properties) {
        return ClusterConfigOptions.newBuilder().setElectionMinTimeout((Integer)properties.get("graphdb.raft.election.min.ms")).setElectionRangeTimeout((Integer)properties.get("graphdb.raft.election.range.ms")).setMessageSize((Integer)properties.get("graphdb.raft.rpc.message.size")).setHeartBeatInterval((Integer)properties.get("graphdb.raft.election.heartbeat.ms")).setVerificationMs((Integer)properties.get("graphdb.raft.rpc.verification.ms")).setBatchUpdateInterval((Integer)properties.get("graphdb.raft.batch.update.interval.ms")).setTransactionLogMaximumSize(((Float)properties.get("graphdb.raft.transaction.log.threshold.gb")).floatValue()).build();
    }

    public static Map<String, Number> collectAllProperties(ClusterConfigOptions properties) {
        HashMap<String, Number> allProperties = new HashMap<String, Number>();
        allProperties.put("graphdb.raft.election.min.ms", properties.getElectionMinTimeout());
        allProperties.put("graphdb.raft.election.range.ms", properties.getElectionRangeTimeout());
        allProperties.put("graphdb.raft.rpc.message.size", properties.getMessageSize());
        allProperties.put("graphdb.raft.rpc.verification.ms", properties.getVerificationMs());
        allProperties.put("graphdb.raft.election.heartbeat.ms", properties.getHeartBeatInterval());
        allProperties.put("graphdb.raft.batch.update.interval.ms", properties.getBatchUpdateInterval());
        allProperties.put("graphdb.raft.transaction.log.threshold.gb", Float.valueOf(properties.getTransactionLogMaximumSize()));
        return allProperties;
    }

    public static void checkForMarkedSnapshots(ClusterGroup clusterGroup) {
        File lockFile;
        File oldSnapshot = clusterGroup.getStateMachine().getLatestSnapshot();
        if (oldSnapshot != null && oldSnapshot.exists() && (lockFile = new File(oldSnapshot.getParent(), "snapshot-delete.lock")).exists()) {
            try {
                String snapshotToDelete = new ObjectMapper().readTree(lockFile).get("snapshotToDelete").asText();
                File snapshotFile = new File(oldSnapshot.getParent(), snapshotToDelete);
                if (snapshotFile.exists()) {
                    snapshotFile.delete();
                }
            }
            catch (IOException e) {
                logger.error("Error reading or processing snapshot lock file: {}", (Object)lockFile.getAbsolutePath(), (Object)e);
            }
            lockFile.delete();
        }
    }

    private static AppendEntry buildSystemHeartbeat(RaftTaskController controller) {
        long lastLogIndex = controller.getLastLogIndex();
        long lastLogTerm = controller.getLastLogTerm();
        return AppendEntry.newBuilder().setChannel(-7).setTerm(controller.getCurrentTerm()).setLeaderId(controller.getCurrentAddress()).setCommitIndex(lastLogIndex).setLogTerm(lastLogTerm).setPrevLogIndex(lastLogIndex).setPrevLogTerm(lastLogTerm).build();
    }

    private static void sendSystemHeartbeat(AppendEntry heartbeat, Quorum quorum, RaftTaskController controller) {
        logger.info("Sending system heartbeat to follower nodes");
        long lastLogIndex = heartbeat.getCommitIndex();
        quorum.increment(true);
        for (RpcNodeClient node : controller.getRpcNodes()) {
            if (node.getStatus().equals((Object)RpcNodeClient.Status.NO_CONNECTION)) {
                throw new RaftRpcConnectionException(String.format("There is no connection with %s", node.getAddress()));
            }
            quorum.addTask(controller.submitTask(() -> {
                AppendResponse response = node.sendTransactionRpc(heartbeat, quorum);
                if (response.getMatchIndex() < lastLogIndex) {
                    logger.warn("Node {} not caught up yet to leader with last log {} compared to leader {}", new Object[]{node.getAddress(), response.getMatchIndex(), lastLogIndex});
                } else if (response.getTerm() != controller.getCurrentTerm()) {
                    logger.warn("Node {} not caught up yet to leader with last log {} compared to leader {}", new Object[]{node.getAddress(), response.getMatchIndex(), lastLogIndex});
                } else if (response.getMatchIndex() > lastLogIndex) {
                    logger.warn("Node {} has higher last log index {} than leader {}", new Object[]{node.getAddress(), response.getMatchIndex(), lastLogIndex});
                    quorum.increment(false);
                } else {
                    logger.info("Node {} is in sync with leader: {}", (Object)node.getAddress(), (Object)response.getSuccess());
                    quorum.increment(response.getSuccess());
                }
            }));
        }
        quorum.awaitAll();
    }

    private static void awaitNodesToSync(@Nullable String node) {
        try {
            if (node == null) {
                logger.info("Waiting for cluster nodes to sync for {}s", (Object)(SYSTEM_HEARTBEAT_TIMEOUT / 1000));
                Thread.sleep(SYSTEM_HEARTBEAT_TIMEOUT);
            } else {
                logger.info("Waiting for node {} to sync for {}s", (Object)node, (Object)(NODE_SYNC_TIMEOUT / 1000));
                Thread.sleep(NODE_SYNC_TIMEOUT);
            }
        }
        catch (InterruptedException e) {
            logger.error("Unable to pass system heartbeat as operation was interrupted");
            Thread.currentThread().interrupt();
            throw new RaftException((Exception)e);
        }
    }

    private static void catchUpNode(Quorum quorum, RpcNodeClient rpcNode, long matchIndex, RaftTaskController state) {
        Future<?> future = state.submitTask(new CatchupTask(rpcNode, state.getLogEntry(matchIndex + 1L), quorum, state));
        if (quorum != null) {
            quorum.addTask(future);
        }
    }
}

