/*
 * Decompiled with CFR 0.152.
 */
package com.ontotext.graphdb.raft.storage.log.persistent;

import com.ontotext.graphdb.raft.grpc.AppendEntry;
import com.ontotext.graphdb.raft.grpc.ConfigEntry;
import com.ontotext.graphdb.raft.grpc.NodeInfo;
import com.ontotext.graphdb.raft.storage.LogChannel;
import com.ontotext.graphdb.raft.storage.LogEntry;
import com.ontotext.graphdb.raft.storage.NoActiveTransactionException;
import com.ontotext.graphdb.raft.storage.TransactionLogException;
import com.ontotext.graphdb.raft.storage.log.persistent.PersistentLogEntry;
import com.ontotext.graphdb.raft.util.RaftUtil;
import java.io.BufferedOutputStream;
import java.io.ByteArrayOutputStream;
import java.io.DataOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.io.RandomAccessFile;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardOpenOption;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Semaphore;
import java.util.function.Function;
import org.apache.commons.io.output.CountingOutputStream;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class PersistentLogChannel
implements LogChannel {
    private static final Logger logger = LoggerFactory.getLogger(PersistentLogChannel.class);
    public static final int INITIAL_OFFSET = 4;
    private final Path channelPath;
    private final Semaphore writerSemaphore;
    private final String repositoryId;
    private final Function<Long, LogEntry> indexedEntrySupplier;
    private final int id;
    private volatile CountingOutputStream counter;
    private volatile DataOutputStream dataStream;
    private volatile LogEntry lastLogEntry;
    private volatile AppendEntry metadata;
    private volatile ConfigEntry configEntryMetadata;
    private volatile long currentPosition;
    private volatile long lastPosition;
    private volatile int fingerprintSize;

    public PersistentLogChannel(String transactionLogDir, String repository, Function<Long, LogEntry> indexedEntrySupplier, int id) throws IOException {
        this.channelPath = Paths.get(transactionLogDir, repository + ".channel");
        this.repositoryId = repository;
        this.indexedEntrySupplier = indexedEntrySupplier;
        this.id = id;
        this.writerSemaphore = new Semaphore(1, true);
        this.currentPosition = -1L;
    }

    public PersistentLogChannel(File log, Function<Long, LogEntry> indexedEntrySupplier) throws IOException {
        this.channelPath = log.toPath();
        String logName = log.getName();
        this.repositoryId = logName.substring(0, logName.length() - ".channel".length());
        this.indexedEntrySupplier = indexedEntrySupplier;
        this.id = this.fetchId();
        this.writerSemaphore = new Semaphore(1, true);
        this.currentPosition = -1L;
    }

    @Override
    public void initialize() throws IOException {
        if (this.currentPosition < 0L) {
            if (!Files.exists(this.channelPath, new LinkOption[0])) {
                logger.info("Creating storage for channel {}", (Object)this.repositoryId);
                File channelFile = this.channelPath.toFile();
                channelFile.getParentFile().mkdirs();
                channelFile.createNewFile();
                this.recordId();
                this.lastLogEntry = null;
                this.currentPosition = 4L;
                this.lastPosition = 4L;
            } else {
                logger.info("Initializing storage for channel {} for repository {}", (Object)this.id, (Object)this.repositoryId);
                this.lastPosition = this.currentPosition = Files.size(this.channelPath);
                if (this.lastPosition > 4L) {
                    this.initLastEntry();
                }
            }
        }
    }

    @Override
    public void shutdown() {
        try {
            this.rollbackEntry();
        }
        catch (Exception e) {
            logger.warn("Error occurred during shutdown of transaction channel {}", (Object)this.repositoryId, (Object)e);
        }
        finally {
            this.lastLogEntry = null;
            this.currentPosition = -1L;
        }
        logger.info("Shut down persistent log channel {}", (Object)this.repositoryId);
    }

    @Override
    public void truncate() {
        try {
            this.shutdown();
            this.channelPath.toFile().delete();
            this.initialize();
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    @Override
    public Path getChannelPath() {
        return this.channelPath;
    }

    @Override
    public int getChannelId() {
        return this.id;
    }

    @Override
    public String getRepositoryId() {
        return this.repositoryId;
    }

    @Override
    public LogEntry getLastLogEntry() {
        return this.lastLogEntry;
    }

    @Override
    public long begin() {
        try {
            this.writerSemaphore.acquire();
            this.clearTransactionVars();
            this.clearConfigUpdateVars();
            this.lastPosition = this.currentPosition;
            FileOutputStream fileStream = new FileOutputStream(this.channelPath.toFile(), true);
            this.counter = new CountingOutputStream((OutputStream)fileStream);
            BufferedOutputStream buffer = new BufferedOutputStream((OutputStream)this.counter);
            this.dataStream = new DataOutputStream(buffer);
            return this.lastPosition;
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            this.dataStream = null;
            throw new TransactionLogException(e);
        }
        catch (IOException e) {
            this.dataStream = null;
            throw new TransactionLogException(e);
        }
    }

    @Override
    public OutputStream getChannelStream() {
        if (this.dataStream == null) {
            throw new NoActiveTransactionException("Channel transaction should be started first before recording updates");
        }
        return this.dataStream;
    }

    @Override
    public void replaceMetadata(AppendEntry metadata) {
        try (RandomAccessFile file = new RandomAccessFile(this.channelPath.toFile(), "rw");){
            this.dataStream.flush();
            file.seek(this.lastPosition);
            ByteBuffer buff = ByteBuffer.wrap(new byte[36]);
            buff.putLong(metadata.getCommitIndex());
            buff.putLong(metadata.getLogTerm());
            buff.putLong(metadata.getPrevLogIndex());
            buff.putLong(metadata.getPrevLogTerm());
            buff.putInt(metadata.getChannel());
            file.write(buff.array());
        }
        catch (IOException e) {
            this.rollbackEntry();
            throw new TransactionLogException(e);
        }
    }

    @Override
    public long getLastOffset() {
        return this.lastPosition;
    }

    @Override
    public void addMetadata(AppendEntry entry) {
        try {
            this.metadata = entry;
            ByteBuffer buff = ByteBuffer.wrap(new byte[36]);
            buff.putLong(entry.getCommitIndex());
            buff.putLong(entry.getLogTerm());
            buff.putLong(entry.getPrevLogIndex());
            buff.putLong(entry.getPrevLogTerm());
            buff.putInt(entry.getChannel());
            this.dataStream.write(buff.array());
        }
        catch (IOException e) {
            this.rollbackEntry();
            throw new TransactionLogException(e);
        }
    }

    @Override
    public void addAffectedChannels(List<String> affectedChannels, boolean clearAll) {
        try {
            ByteArrayOutputStream stream = new ByteArrayOutputStream();
            DataOutputStream wrappedStream = new DataOutputStream(stream);
            for (String channel : affectedChannels) {
                byte[] bytes = channel.getBytes(StandardCharsets.UTF_8);
                wrappedStream.writeInt(bytes.length);
                wrappedStream.write(bytes);
            }
            byte[] data = stream.toByteArray();
            this.dataStream.write(clearAll ? 1 : 0);
            this.dataStream.writeInt(data.length);
            this.dataStream.write(data);
        }
        catch (IOException e) {
            this.rollbackEntry();
            throw new TransactionLogException(e);
        }
    }

    @Override
    public void addData(AppendEntry data) {
        try {
            this.dataStream.write(data.getRdfData().toByteArray());
        }
        catch (IOException e) {
            this.rollbackEntry();
            throw new TransactionLogException(e);
        }
    }

    @Override
    public void addMetadata(ConfigEntry entry) {
        try {
            this.configEntryMetadata = entry;
            ByteBuffer buff = ByteBuffer.wrap(new byte[36]);
            buff.putLong(entry.getCommitIndex());
            buff.putLong(entry.getLogTerm());
            buff.putLong(entry.getPrevLogIndex());
            buff.putLong(entry.getPrevLogTerm());
            buff.putInt(entry.getChannel());
            this.dataStream.write(buff.array());
        }
        catch (IOException e) {
            this.rollbackEntry();
            throw new TransactionLogException(e);
        }
    }

    @Override
    public void addData(ConfigEntry data) {
        try {
            if (!data.getNewConfigServersList().isEmpty()) {
                this.dataStream.writeInt(0);
                this.addAddressesInDataStream(data.getNewConfigServersList());
                this.addAddressesInDataStream(data.getOldConfigList());
            } else {
                this.dataStream.writeInt(1);
                this.addGroupPropertiesInDataStream(RaftUtil.collectAllProperties(data.getProperties()));
            }
        }
        catch (IOException e) {
            this.rollbackEntry();
            throw new TransactionLogException(e);
        }
    }

    private void addAddressesInDataStream(List<NodeInfo> nodes) throws IOException {
        this.dataStream.writeInt(nodes.size());
        for (NodeInfo nodeInfo : nodes) {
            Object combinedAddress = nodeInfo.getRpcAddress();
            if (StringUtils.isNotEmpty((CharSequence)nodeInfo.getHttpAddress())) {
                combinedAddress = (String)combinedAddress + ";" + nodeInfo.getHttpAddress();
            }
            this.dataStream.writeInt(((String)combinedAddress).getBytes(StandardCharsets.UTF_8).length);
            this.dataStream.write(((String)combinedAddress).getBytes(StandardCharsets.UTF_8));
        }
    }

    private void addGroupPropertiesInDataStream(Map<String, Number> propertiesMap) throws IOException {
        this.dataStream.writeInt(propertiesMap.size());
        for (String property : propertiesMap.keySet()) {
            byte[] bytes = property.getBytes(StandardCharsets.UTF_8);
            this.dataStream.writeInt(bytes.length);
            this.dataStream.write(bytes);
            if ("graphdb.raft.transaction.log.threshold.gb".equals(property)) {
                this.dataStream.writeFloat(((Float)propertiesMap.get(property)).floatValue());
                continue;
            }
            this.dataStream.writeInt((Integer)propertiesMap.get(property));
        }
    }

    @Override
    public void updateLastLog(LogEntry lastLog) {
        this.lastLogEntry = lastLog;
    }

    @Override
    public void addData(byte[] data, int length) {
        try {
            this.dataStream.write(data, 0, length);
        }
        catch (IOException e) {
            this.rollbackEntry();
            throw new TransactionLogException(e);
        }
    }

    @Override
    public int addFingerprint(String fingerprint) {
        try {
            if (this.dataStream == null) {
                throw new NoActiveTransactionException("No active transaction for repository " + this.getRepositoryId());
            }
            byte[] bytes = fingerprint.getBytes(StandardCharsets.UTF_8);
            this.dataStream.write(bytes);
            this.fingerprintSize = bytes.length;
            return this.fingerprintSize;
        }
        catch (IOException e) {
            this.rollbackEntry();
            throw new TransactionLogException(e);
        }
    }

    @Override
    public long commitEntry() {
        try {
            this.validateTransactionVars();
            this.dataStream.writeLong(this.lastPosition);
            this.dataStream.flush();
            this.dataStream.close();
            this.currentPosition += this.counter.getByteCount();
            this.lastPosition = this.currentPosition;
            this.clearTransactionVars();
            long l = this.lastPosition;
            return l;
        }
        catch (Exception e) {
            this.rollbackEntry();
            throw new TransactionLogException(e);
        }
        finally {
            this.writerSemaphore.release();
        }
    }

    @Override
    public long commitConfigEntry() {
        try {
            this.validateConfigUpdateVars();
            this.dataStream.writeLong(this.lastPosition);
            this.dataStream.flush();
            this.dataStream.close();
            this.currentPosition += this.counter.getByteCount();
            this.lastPosition = this.currentPosition;
            this.clearConfigUpdateVars();
            long l = this.lastPosition;
            return l;
        }
        catch (Exception e) {
            this.rollbackEntry();
            throw new TransactionLogException(e);
        }
        finally {
            this.writerSemaphore.release();
        }
    }

    @Override
    public void deleteChannel() {
        try {
            this.shutdown();
            Files.delete(this.channelPath);
        }
        catch (Exception e) {
            throw new TransactionLogException("Unable to delete channel", e);
        }
    }

    @Override
    public LogEntry getLogEntry(long position, long size, int fingerprintSize, long index, long term, LogEntry.Status initStatus) {
        if (position + size > this.currentPosition) {
            throw new IndexOutOfBoundsException("No transaction log at position " + position);
        }
        return new PersistentLogEntry(this, initStatus, position, size, fingerprintSize, index, term);
    }

    @Override
    public boolean rollbackEntry() {
        try {
            if (this.dataStream != null) {
                logger.info("Rolling back entry on channel {}", (Object)this.repositoryId);
                this.dataStream.close();
                boolean bl = true;
                return bl;
            }
            boolean bl = false;
            return bl;
        }
        catch (IOException e) {
            throw new TransactionLogException("Unable to close channel data stream", e);
        }
        finally {
            this.clearTransactionVars();
            this.truncateChannel(this.lastPosition);
        }
    }

    private int fetchId() throws IOException {
        try (RandomAccessFile file = new RandomAccessFile(this.channelPath.toFile(), "r");){
            int n = file.readInt();
            return n;
        }
    }

    private void recordId() throws IOException {
        try (RandomAccessFile file = new RandomAccessFile(this.channelPath.toFile(), "rw");){
            file.writeInt(this.id);
        }
    }

    private void initLastEntry() {
        this.lastLogEntry = this.indexedEntrySupplier.apply(this.getLastEntryIndex());
    }

    private void truncateChannel(long pos) {
        try {
            FileChannel.open(this.channelPath, StandardOpenOption.WRITE).truncate(pos).close();
            this.lastPosition = pos;
            this.currentPosition = pos;
        }
        catch (IOException e) {
            throw new TransactionLogException("Unable to rollback last transaction log", e);
        }
        finally {
            this.writerSemaphore.release();
        }
    }

    private long getLastEntryIndex() {
        return this.getEntryIndex(this.lastPosition);
    }

    private long getEntryIndex(long pos) {
        long l;
        RandomAccessFile channel = new RandomAccessFile(this.channelPath.toFile(), "r");
        try {
            channel.seek(pos - 8L);
            long lastLogOffset = channel.readLong();
            channel.seek(lastLogOffset);
            l = channel.readLong();
        }
        catch (Throwable throwable) {
            try {
                try {
                    channel.close();
                }
                catch (Throwable throwable2) {
                    throwable.addSuppressed(throwable2);
                }
                throw throwable;
            }
            catch (IOException e) {
                throw new TransactionLogException("Unable to fetch last entry index", e);
            }
        }
        channel.close();
        return l;
    }

    private void validateTransactionVars() {
        if (this.metadata == null || this.fingerprintSize < 1 || this.dataStream == null || this.counter == null) {
            throw new TransactionLogException("Unable to commit transaction");
        }
    }

    private void validateConfigUpdateVars() {
        if (this.configEntryMetadata == null || this.fingerprintSize < 1 || this.counter == null) {
            throw new TransactionLogException("Unable to commit config update");
        }
    }

    private void clearTransactionVars() {
        this.metadata = null;
        this.fingerprintSize = -1;
        this.dataStream = null;
        this.counter = null;
    }

    private void clearConfigUpdateVars() {
        this.configEntryMetadata = null;
        this.fingerprintSize = -1;
        this.dataStream = null;
        this.counter = null;
    }
}

