/*
 * Decompiled with CFR 0.152.
 */
package com.ontotext.graphdb.raft.storage.log.persistent;

import com.google.common.annotations.VisibleForTesting;
import com.google.protobuf.ByteString;
import com.ontotext.graphdb.raft.node.concurrent.SemaphoreLock;
import com.ontotext.graphdb.raft.storage.LogChannel;
import com.ontotext.graphdb.raft.storage.LogEntry;
import com.ontotext.graphdb.raft.storage.TransactionLogException;
import com.ontotext.graphdb.raft.storage.log.memory.InMemoryLogEntry;
import com.ontotext.graphdb.raft.storage.log.persistent.PersistentLogEntry;
import java.io.EOFException;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.FileAttribute;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.locks.Lock;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class LogIndex {
    private static final Logger logger = LoggerFactory.getLogger(LogIndex.class);
    public static final String INDEX_FILE = "log.index";
    private static final int BYTES_PER_ENTRY = 33;
    private static final int STATUS_OFFSET = 32;
    private final Path logIndexPath;
    private final Map<Integer, LogChannel> channelMap;
    private final Lock indexLock;
    private final AtomicLong lastValidLog;
    private final AtomicInteger repositoryIndex;
    private final Map<Long, PersistentLogEntry> lastLogEntries;
    private volatile LogEntry lastLogEntry;
    private volatile long startingIndex;

    public LogIndex(String transactionLogDir, Map<Integer, LogChannel> channelMap, Map<Long, PersistentLogEntry> lastLogEntries) {
        this.logIndexPath = Paths.get(transactionLogDir, INDEX_FILE);
        this.channelMap = channelMap;
        this.lastLogEntries = lastLogEntries;
        this.indexLock = new SemaphoreLock();
        this.lastValidLog = new AtomicLong(0L);
        this.repositoryIndex = new AtomicInteger(-1);
        this.startingIndex = -1L;
    }

    public void initialize(long lastValidLog) {
        try {
            if (Files.exists(this.logIndexPath, new LinkOption[0])) {
                long lastLogIndex = this.calculateLastIndex();
                logger.info("Initializing persistent log index. Last log set to {}", (Object)lastLogIndex);
                this.channelMap.get(-6).getLastLogEntry();
                if (lastLogIndex > this.startingIndex) {
                    this.lastLogEntry = this.fetchLogEntryFromStorage(lastLogIndex);
                    this.lastValidLog.set(lastValidLog);
                } else {
                    this.lastLogEntries.clear();
                    this.lastLogEntry = null;
                }
            } else {
                logger.info("Initializing persistent log index for the first time");
                this.createLogIndexFile();
                this.lastLogEntry = null;
            }
            int fetchedRepositoryIndex = this.fetchRepositoryIndex();
            logger.debug("Last known repository index is {}", (Object)fetchedRepositoryIndex);
        }
        catch (IOException e) {
            logger.error("Log Index initialization failed. Setting last valid log to 0.");
            this.lastValidLog.set(0L);
            throw new TransactionLogException(e);
        }
    }

    public void shutdown() {
        boolean locked = false;
        try {
            locked = this.indexLock.tryLock(30L, TimeUnit.SECONDS);
        }
        catch (InterruptedException e) {
            logger.error("Interrupted during Log Index shutdown", (Throwable)e);
            Thread.currentThread().interrupt();
        }
        finally {
            this.repositoryIndex.set(-1);
            this.lastLogEntry = null;
            if (locked) {
                this.indexLock.unlock();
            }
        }
        logger.info("Shut down Log Index");
    }

    public void truncate(long lastValidLog) {
        this.shutdown();
        this.lastLogEntries.clear();
        try {
            int lastKnownRepositoryIndex = this.getRepositoryIndexUnsafe();
            Files.delete(this.logIndexPath);
            this.createLogIndexFile();
            this.setRepositoryIndex(lastKnownRepositoryIndex);
            this.setStartingIndex(lastValidLog - 1L);
        }
        catch (IOException e) {
            logger.error("Log Index truncate operation failed. Setting last valid log to 0.");
            this.lastValidLog.set(0L);
            throw new TransactionLogException("Failed to truncate log index", e);
        }
        this.initialize(lastValidLog);
    }

    public long getLastLogIndex() {
        if (this.lastLogEntry != null) {
            return this.lastLogEntry.getIndex();
        }
        return 0L;
    }

    public long getStartingIndex() {
        return this.startingIndex;
    }

    public LogEntry getLastLog() {
        return this.lastLogEntry;
    }

    public long getLastValidLog() {
        return this.lastValidLog.get();
    }

    public long getLastLogTerm() {
        if (this.lastLogEntry != null) {
            return this.lastLogEntry.getTerm();
        }
        return 0L;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public LogEntry fetchLogEntry(long logIndex) {
        this.indexLock.lock();
        try {
            PersistentLogEntry lastLog;
            if (logIndex <= this.startingIndex) {
                LogEntry logEntry = null;
                return logEntry;
            }
            LogEntry lastLogEntry = this.lastLogEntry;
            if (lastLogEntry != null) {
                if (logIndex > lastLogEntry.getIndex()) {
                    LogEntry logEntry = null;
                    return logEntry;
                }
                if (logIndex == lastLogEntry.getIndex() && this.channelMap.containsKey(lastLogEntry.getChannel())) {
                    LogEntry logEntry = lastLogEntry;
                    return logEntry;
                }
            }
            if ((lastLog = this.lastLogEntries.get(logIndex)) != null && lastLog.getIndex() == logIndex) {
                PersistentLogEntry persistentLogEntry = lastLog;
                return persistentLogEntry;
            }
            LogEntry logEntry = this.fetchLogEntryFromStorageUnsafe(logIndex);
            return logEntry;
        }
        finally {
            this.indexLock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public LogEntry fetchLogEntryFromStorage(long logIndex) {
        this.indexLock.lock();
        try {
            LogEntry logEntry = this.fetchLogEntryFromStorageUnsafe(logIndex);
            return logEntry;
        }
        finally {
            this.indexLock.unlock();
        }
    }

    public LogEntry.Status fetchLogEntryStatus(long logIndex) {
        LogEntry.Status status;
        PersistentLogEntry lastLog;
        LogEntry lastLogEntry = this.lastLogEntry;
        if (logIndex <= this.startingIndex) {
            return null;
        }
        if (lastLogEntry != null) {
            if (logIndex > lastLogEntry.getIndex()) {
                return null;
            }
            if (logIndex == lastLogEntry.getIndex() && this.channelMap.containsKey(lastLogEntry.getChannel())) {
                return lastLogEntry.getStatus();
            }
        }
        if ((lastLog = this.lastLogEntries.get(logIndex)) != null && lastLog.getIndex() == logIndex) {
            return lastLog.getStatus();
        }
        RandomAccessFile logIndexStorage = new RandomAccessFile(this.logIndexPath.toFile(), "r");
        try {
            logIndexStorage.seek(this.getEntryPosition(logIndex) + 32L);
            status = LogEntry.Status.fromCode(logIndexStorage.readByte());
        }
        catch (Throwable throwable) {
            try {
                try {
                    logIndexStorage.close();
                }
                catch (Throwable throwable2) {
                    throwable.addSuppressed(throwable2);
                }
                throw throwable;
            }
            catch (IOException e) {
                throw new TransactionLogException("Could not retrieve log with id " + logIndex, e);
            }
        }
        logIndexStorage.close();
        return status;
    }

    public LogEntry appendLogEntryData(int channel, long pos, long length, long term, int fingerprintSize, LogEntry.Status status) {
        this.indexLock.lock();
        long newLastLogIndex = this.lastLogEntry == null ? 1L + this.fetchStartingIndex() : this.lastLogEntry.getIndex() + 1L;
        try {
            LogEntry logEntry;
            RandomAccessFile logIndexStorage = new RandomAccessFile(this.logIndexPath.toFile(), "rw");
            try {
                logIndexStorage.seek(this.getEntryPosition(newLastLogIndex));
                ByteBuffer buff = ByteBuffer.wrap(new byte[33]);
                buff.putInt(channel);
                buff.putLong(pos);
                buff.putLong(length);
                buff.putLong(term);
                buff.putInt(fingerprintSize);
                buff.put(status.getStatusCode());
                logIndexStorage.write(buff.array());
                LogEntry entry = channel != -4 ? new PersistentLogEntry(this.channelMap.get(channel), status, pos, length, fingerprintSize, newLastLogIndex, term) : this.fetchEmptyLog(newLastLogIndex, term, status);
                this.lastLogEntry = entry;
                logEntry = entry;
            }
            catch (Throwable throwable) {
                try {
                    try {
                        logIndexStorage.close();
                    }
                    catch (Throwable throwable2) {
                        throwable.addSuppressed(throwable2);
                    }
                    throw throwable;
                }
                catch (IOException e) {
                    throw new TransactionLogException("Could not record log with id " + newLastLogIndex, e);
                }
            }
            logIndexStorage.close();
            return logEntry;
        }
        finally {
            this.indexLock.unlock();
        }
    }

    public void setLogStatus(long entryId, LogEntry.Status newStatus) {
        PersistentLogEntry lastLog;
        this.indexLock.lock();
        if (this.lastLogEntry != null && entryId == this.lastLogEntry.getIndex()) {
            this.lastLogEntry.setStatus(newStatus);
        }
        if ((lastLog = this.lastLogEntries.get(entryId)) != null && this.lastLogEntry != lastLog && lastLog.getIndex() == entryId) {
            lastLog.setStatus(newStatus);
        }
        try (RandomAccessFile logIndexStorage = new RandomAccessFile(this.logIndexPath.toFile(), "rw");){
            logIndexStorage.seek(this.getEntryPosition(entryId) + 32L);
            logIndexStorage.writeByte(newStatus.getStatusCode());
            if (newStatus == LogEntry.Status.VALID && this.lastValidLog.get() < entryId) {
                this.lastValidLog.set(entryId);
            }
        }
        catch (IOException e) {
            throw new TransactionLogException("Could not update status for log with id " + entryId, e);
        }
        finally {
            this.indexLock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public int incrementAndGetRepositoryIndex() {
        this.indexLock.lock();
        try {
            int currentIndex = this.getRepositoryIndexUnsafe();
            int nextIndex = currentIndex + 1;
            this.setRepositoryIndex(nextIndex);
            int n = nextIndex;
            return n;
        }
        finally {
            this.indexLock.unlock();
        }
    }

    public void rollbackUpTo(long entryId) {
        this.indexLock.lock();
        try (FileChannel channel = FileChannel.open(this.logIndexPath, StandardOpenOption.WRITE);){
            channel.truncate(this.getEntryPosition(entryId));
        }
        catch (IOException e) {
            throw new TransactionLogException("Unable to rollback last transaction log", e);
        }
        finally {
            this.indexLock.unlock();
        }
    }

    @VisibleForTesting
    public long calculateLastIndex() {
        try {
            if (Files.exists(this.logIndexPath, new LinkOption[0])) {
                long calculatedIndexCount = Files.size(this.logIndexPath) / 33L;
                return calculatedIndexCount <= 0L && this.fetchStartingIndex() != 0L ? this.fetchStartingIndex() : calculatedIndexCount - 1L + this.fetchStartingIndex();
            }
        }
        catch (IOException e) {
            logger.error("Error while calculating last index", (Throwable)e);
        }
        return -1L;
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private LogEntry fetchLogEntryFromStorageUnsafe(long logIndex) {
        try (RandomAccessFile logIndexStorage = new RandomAccessFile(this.logIndexPath.toFile(), "r");){
            logIndexStorage.seek(this.getEntryPosition(logIndex));
            ByteBuffer buff = ByteBuffer.wrap(new byte[33]);
            logIndexStorage.read(buff.array());
            int channel = buff.getInt();
            long offset = buff.getLong();
            long length = buff.getLong();
            long term = buff.getLong();
            int fingerprintSize = buff.getInt();
            LogEntry.Status status = LogEntry.Status.fromCode(buff.get());
            LogChannel persistentChannel = this.channelMap.get(channel);
            if (persistentChannel == null) {
                LogEntry logEntry = this.fetchEmptyLog(logIndex, term, status);
                return logEntry;
            }
            LogEntry logEntry = this.fetchLogFromChannel(persistentChannel, offset, length, fingerprintSize, logIndex, term, status);
            return logEntry;
        }
        catch (IOException e) {
            throw new TransactionLogException("Could not retrieve log with id " + logIndex, e);
        }
    }

    private void setRepositoryIndex(int repositoryIndex) {
        int lastRepositoryIndex = this.fetchRepositoryIndex();
        if (lastRepositoryIndex > repositoryIndex) {
            throw new TransactionLogException("Cannot replace current repository index " + lastRepositoryIndex + " with lower index " + repositoryIndex);
        }
        try (RandomAccessFile logIndexStorage = new RandomAccessFile(this.logIndexPath.toFile(), "rw");){
            logIndexStorage.seek(0L);
            logIndexStorage.writeInt(repositoryIndex);
            this.repositoryIndex.set(repositoryIndex);
        }
        catch (IOException e) {
            throw new TransactionLogException("Could not update repository index", e);
        }
    }

    private void setStartingIndex(long startingIndex) {
        if (this.startingIndex > startingIndex) {
            throw new TransactionLogException("Cannot replace current starting index " + this.startingIndex + " with lower index " + startingIndex);
        }
        try (RandomAccessFile logIndexStorage = new RandomAccessFile(this.logIndexPath.toFile(), "rw");){
            logIndexStorage.seek(4L);
            logIndexStorage.writeLong(startingIndex);
            this.startingIndex = startingIndex;
        }
        catch (IOException e) {
            throw new TransactionLogException("Could not update repository index", e);
        }
    }

    private int fetchRepositoryIndex() {
        if (this.repositoryIndex.get() < 0) {
            this.repositoryIndex.set(this.getRepositoryIndexUnsafe());
        }
        return this.repositoryIndex.get();
    }

    private int getRepositoryIndexUnsafe() {
        int n;
        RandomAccessFile logIndexStorage = new RandomAccessFile(this.logIndexPath.toFile(), "rw");
        try {
            logIndexStorage.seek(0L);
            int index = logIndexStorage.readInt();
            n = Math.max(index, 0);
        }
        catch (Throwable throwable) {
            try {
                try {
                    logIndexStorage.close();
                }
                catch (Throwable throwable2) {
                    throwable.addSuppressed(throwable2);
                }
                throw throwable;
            }
            catch (EOFException ex) {
                return 0;
            }
            catch (IOException e) {
                throw new TransactionLogException("Could not read repository index from storage", e);
            }
        }
        logIndexStorage.close();
        return n;
    }

    public int getRepositoryIndex() {
        return this.fetchRepositoryIndex();
    }

    public void restoreRepositoryIndex(int repositoryIndex) {
        this.indexLock.lock();
        try {
            this.repositoryIndex.set(repositoryIndex);
            this.setRepositoryIndex(repositoryIndex);
        }
        finally {
            this.indexLock.unlock();
        }
    }

    private long fetchStartingIndex() {
        if (this.startingIndex < 0L) {
            try (RandomAccessFile logIndexStorage = new RandomAccessFile(this.logIndexPath.toFile(), "rw");){
                logIndexStorage.seek(4L);
                long index = logIndexStorage.readLong();
                this.startingIndex = Math.max(index, 0L);
            }
            catch (EOFException ex) {
                this.startingIndex = 0L;
            }
            catch (IOException e) {
                throw new TransactionLogException("Could not read starting index from storage", e);
            }
        }
        return this.startingIndex;
    }

    private LogEntry fetchLogFromChannel(LogChannel channel, long offset, long length, int fingerprintSize, long entryId, long term, LogEntry.Status status) {
        return channel.getLogEntry(offset, length, fingerprintSize, entryId, term, status);
    }

    private LogEntry fetchEmptyLog(long logIndex, long term, LogEntry.Status status) {
        InMemoryLogEntry entry = new InMemoryLogEntry(logIndex, term, ByteString.EMPTY, "", -4, 33L);
        entry.setStatus(status);
        return entry;
    }

    private void createLogIndexFile() throws IOException {
        Files.createFile(this.logIndexPath, new FileAttribute[0]);
    }

    private long getEntryPosition(long index) {
        return (index - this.fetchStartingIndex()) * 33L;
    }
}

