/*
 * Decompiled with CFR 0.152.
 */
package com.ontotext.graphdb.raft.node.task;

import com.google.protobuf.ByteString;
import com.ontotext.graphdb.raft.NodeState;
import com.ontotext.graphdb.raft.RaftException;
import com.ontotext.graphdb.raft.grpc.AppendEntry;
import com.ontotext.graphdb.raft.grpc.RpcNodeClient;
import com.ontotext.graphdb.raft.node.Quorum;
import com.ontotext.graphdb.raft.node.RaftTaskController;
import com.ontotext.graphdb.raft.node.task.AppendEntryTask;
import com.ontotext.graphdb.raft.storage.LogEntry;
import com.ontotext.graphdb.raft.storage.TransactionLog;
import com.ontotext.graphdb.raft.util.RaftUtil;
import java.util.Map;
import java.util.Set;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class TruncateLogTask
implements Runnable {
    private static final Logger logger = LoggerFactory.getLogger(TruncateLogTask.class);
    private final Quorum quorum;
    private final RaftTaskController controller;
    private final Consumer<LogEntry> quorumVerification;
    private final Consumer<LogEntry> heartbeatEntry;

    public TruncateLogTask(Quorum quorum, RaftTaskController controller, Consumer<LogEntry> quorumVerification, Consumer<LogEntry> heartbeatEntry) {
        this.quorum = quorum;
        this.controller = controller;
        this.quorumVerification = quorumVerification;
        this.heartbeatEntry = heartbeatEntry;
    }

    @Override
    public void run() {
        this.validateNodeState();
        this.lockInstance();
        try {
            LogEntry entry;
            this.validateNodeState();
            this.awaitValidSecondaryTagsState();
            if (!RaftUtil.isClusterInSync(this.quorum, this.controller)) {
                throw new RaftException("Unable to truncate log as nodes were unable to sync");
            }
            AppendEntry metadata = this.buildMetadata();
            this.heartbeatEntry.accept(this.controller.getTransactionLog().getLastLog());
            try {
                entry = this.processTruncateOnMachine(metadata);
                this.sendTruncateEntryToFollowers(metadata);
                this.quorum.awaitStart();
            }
            finally {
                this.heartbeatEntry.accept(null);
            }
            this.quorum.await();
            this.verifyTruncateEntry(entry);
        }
        catch (Exception e) {
            logger.error("Unlocking transaction log as error occurred during truncate operation: ", (Throwable)e);
            this.unlockFailedInstance();
            throw e;
        }
    }

    private void lockInstance() {
        this.controller.getTransactionLog().beginTruncate();
    }

    private void unlockFailedInstance() {
        this.controller.getTransactionLog().rollbackTruncate();
    }

    private boolean isNodeEmpty() {
        return this.controller.getTransactionLog().getLastLogIndex() < 1L && this.controller.getLatestSnapshotIndex() < 1L;
    }

    private void validateNodeState() {
        NodeState state = this.controller.getNodeState();
        if (state != NodeState.LEADER) {
            throw new RaftException("Unable to truncate log as node is in state " + String.valueOf((Object)state));
        }
        if (this.isNodeEmpty()) {
            throw new RaftException("Unable to truncate log as node is empty");
        }
    }

    private void awaitValidSecondaryTagsState() {
        Set<Map.Entry<String, Long>> entrySet = this.controller.getGroup().getTagMap().entrySet();
        for (int attempts = 15; entrySet.stream().anyMatch(x -> (Long)x.getValue() < this.controller.getLastLogIndex()) && attempts > 0; --attempts) {
            try {
                Thread.sleep(1000L);
                continue;
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                throw new RuntimeException(e);
            }
        }
        for (Map.Entry<String, Long> tag : entrySet) {
            if (tag.getValue() >= this.controller.getLastLogIndex()) continue;
            throw new RaftException("Unable to truncate log as secondary cluster with tag " + tag.getKey() + " and match index " + String.valueOf(tag.getValue()) + " has not caught up with primary cluster index " + this.controller.getLastLogIndex());
        }
    }

    private LogEntry processTruncateOnMachine(AppendEntry metadata) {
        try {
            LogEntry truncateEntry = this.controller.getTransactionLog().commitTruncate(metadata);
            logger.info("Successfully truncated log channels locally up to index excl. {}", (Object)truncateEntry.getIndex());
            this.controller.applyTruncateToMachine(truncateEntry);
            logger.info("Cleaned up local cached recovery snapshots");
            this.quorum.increment(true);
            return truncateEntry;
        }
        catch (Exception e) {
            this.controller.goOutOfSync();
            throw e;
        }
    }

    private AppendEntry buildMetadata() {
        TransactionLog transactionLog = this.controller.getTransactionLog();
        long lastLogIndex = transactionLog.getLastLogIndex();
        long lastLogTerm = transactionLog.getLastLogTerm();
        long term = this.controller.getCurrentTerm();
        return AppendEntry.newBuilder().setFingerprint("" + lastLogIndex).setRdfData(ByteString.copyFromUtf8((String)("" + lastLogIndex))).setCommitIndex(transactionLog.getNextLogIndex()).setLeaderId(this.controller.getCurrentAddress()).setLogTerm(term).setPrevLogIndex(lastLogIndex).setPrevLogTerm(lastLogTerm).setTerm(term).setChannel(-6).build();
    }

    private void sendTruncateEntryToFollowers(AppendEntry metadata) {
        for (RpcNodeClient node : this.controller.getRpcNodes()) {
            this.quorum.addTask(this.controller.submitTask(new AppendEntryTask(node, metadata, null, this.quorum, this.controller)));
        }
    }

    private void verifyTruncateEntry(LogEntry entry) {
        if (this.quorum.getState() == Quorum.State.SUCCESSFUL) {
            logger.info("Truncate entry {} successfully validated with followers", (Object)entry.getIndex());
            this.controller.getTransactionLog().validateEntry(entry);
        } else if (this.quorum.getState() == Quorum.State.NOT_REACHED) {
            logger.info("Truncate entry {} is still being validated by quorum", (Object)entry.getIndex());
            this.quorumVerification.accept(entry);
            if (entry.getStatus() != LogEntry.Status.VALID) {
                this.goOutOfSync(entry);
            }
        } else {
            this.goOutOfSync(entry);
        }
    }

    private void goOutOfSync(LogEntry entry) {
        logger.info("Truncate entry {} is not validated by quorum", (Object)entry.getIndex());
        this.controller.goOutOfSync();
    }
}

