/*
 * Decompiled with CFR 0.152.
 */
package pitt.search.semanticvectors.vectors;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.ClosedByInterruptException;
import java.util.Random;
import java.util.logging.Logger;
import org.apache.lucene.store.IndexInput;
import org.apache.lucene.store.IndexOutput;
import org.eclipse.rdf4j.query.QueryInterruptedException;
import pitt.search.semanticvectors.vectors.CircleLookupTable;
import pitt.search.semanticvectors.vectors.ComplexVectorUtils;
import pitt.search.semanticvectors.vectors.IncompatibleVectorsException;
import pitt.search.semanticvectors.vectors.Vector;
import pitt.search.semanticvectors.vectors.VectorType;

public class ComplexVector
implements Vector {
    public static final Logger logger = Logger.getLogger(ComplexVector.class.getCanonicalName());
    private static Mode DOMINANT_MODE = Mode.CARTESIAN;
    private final int dimension;
    private float[] coordinates;
    private short[] phaseAngles;
    private short[] sparseOffsets;
    private Mode opMode;

    @Override
    public VectorType getVectorType() {
        return VectorType.COMPLEX;
    }

    public static void setDominantMode(Mode mode) {
        if (DOMINANT_MODE == mode) {
            return;
        }
        if (mode == Mode.POLAR_SPARSE) {
            throw new IllegalArgumentException("POLAR_SPARSE cannot be used as dominant mode.");
        }
        logger.info("Globally setting complex DOMINANT_MODE to: '" + (Object)((Object)mode) + "'");
        DOMINANT_MODE = mode;
    }

    public static Mode getDominantMode() {
        return DOMINANT_MODE;
    }

    protected ComplexVector(int dimension, Mode opMode) {
        this.opMode = opMode;
        this.dimension = dimension;
        switch (opMode) {
            case POLAR_SPARSE: {
                this.sparseOffsets = new short[0];
                return;
            }
            case POLAR_DENSE: {
                this.phaseAngles = new short[dimension];
                for (int i = 0; i < dimension; ++i) {
                    this.phaseAngles[i] = -1;
                }
            }
            case CARTESIAN: {
                this.coordinates = new float[2 * dimension];
            }
            case HERMITIAN: {
                this.coordinates = new float[2 * dimension];
            }
        }
    }

    @Override
    public ComplexVector copy() {
        ComplexVector copy = new ComplexVector(this.dimension, this.opMode);
        switch (this.opMode) {
            case POLAR_SPARSE: {
                copy.sparseOffsets = new short[this.sparseOffsets.length];
                for (int i = 0; i < this.sparseOffsets.length; ++i) {
                    copy.sparseOffsets[i] = this.sparseOffsets[i];
                }
                copy.opMode = Mode.POLAR_SPARSE;
                break;
            }
            case POLAR_DENSE: {
                for (int i = 0; i < this.dimension; ++i) {
                    copy.phaseAngles[i] = this.phaseAngles[i];
                }
                break;
            }
            case CARTESIAN: {
                for (int i = 0; i < 2 * this.dimension; ++i) {
                    copy.coordinates[i] = this.coordinates[i];
                }
                break;
            }
            case HERMITIAN: {
                for (int i = 0; i < 2 * this.dimension; ++i) {
                    copy.coordinates[i] = this.coordinates[i];
                }
                break;
            }
        }
        return copy;
    }

    @Override
    public String toString() {
        StringBuilder debugString = new StringBuilder("ComplexVector.");
        switch (this.opMode) {
            case POLAR_SPARSE: {
                debugString.append("  Sparse polar.  Offsets are:\n");
                for (short sparseOffset : this.sparseOffsets) {
                    debugString.append(sparseOffset).append(" ");
                }
                debugString.append("\n");
                break;
            }
            case POLAR_DENSE: {
                debugString.append("  Dense polar. Coordinates are:\n");
                for (short coordinate : this.phaseAngles) {
                    debugString.append(coordinate).append(" ");
                }
                debugString.append("\n");
                break;
            }
            case CARTESIAN: {
                debugString.append("  Cartesian. Coordinates are:\n");
                for (float coordinate : this.coordinates) {
                    debugString.append(coordinate).append(" ");
                }
                debugString.append("\n");
                break;
            }
            case HERMITIAN: {
                debugString.append("  Hermitian. Coordinates are:\n");
                for (float coordinate : this.coordinates) {
                    debugString.append(coordinate).append(" ");
                }
                debugString.append("\n");
            }
        }
        return debugString.toString();
    }

    @Override
    public boolean isZeroVector() {
        switch (this.opMode) {
            case POLAR_SPARSE: {
                return this.sparseOffsets == null || this.sparseOffsets.length == 0;
            }
            case POLAR_DENSE: {
                return this.phaseAngles == null;
            }
            case CARTESIAN: {
                if (this.coordinates == null) {
                    return true;
                }
                for (float coordinate : this.coordinates) {
                    if (coordinate == 0.0f) continue;
                    return false;
                }
                return true;
            }
            case HERMITIAN: {
                if (this.coordinates == null) {
                    return true;
                }
                for (float coordinate : this.coordinates) {
                    if (coordinate == 0.0f) continue;
                    return false;
                }
                return true;
            }
        }
        throw new IllegalArgumentException("Unrecognized mode: " + (Object)((Object)this.opMode));
    }

    @Override
    public ComplexVector generateRandomVector(int dimension, int numEntries, Random random) {
        if (dimension == numEntries) {
            return this.generateRandomVector(dimension, random);
        }
        ComplexVector randomVector = new ComplexVector(dimension, Mode.POLAR_SPARSE);
        boolean[] occupiedPositions = new boolean[dimension];
        randomVector.sparseOffsets = new short[numEntries * 2];
        int entryCount = 0;
        while (entryCount < numEntries) {
            int testPlace = random.nextInt(dimension);
            short randomPhaseAngle = (short)random.nextInt(16384);
            if (occupiedPositions[testPlace]) continue;
            int offsetIdx = entryCount << 1;
            occupiedPositions[testPlace] = true;
            randomVector.sparseOffsets[offsetIdx] = (short)testPlace;
            randomVector.sparseOffsets[offsetIdx + 1] = randomPhaseAngle;
            ++entryCount;
        }
        return randomVector;
    }

    public ComplexVector generateRandomVector(int dimension, Random random) {
        if (ComplexVector.getDominantMode().equals((Object)Mode.HERMITIAN)) {
            return this.generateHermitianRandomVector(dimension, random);
        }
        ComplexVector randomVector = new ComplexVector(dimension, Mode.POLAR_DENSE);
        for (int d = 0; d < randomVector.phaseAngles.length; ++d) {
            randomVector.phaseAngles[d] = (short)random.nextInt(16384);
        }
        return randomVector;
    }

    public ComplexVector generateHermitianRandomVector(int dimension, Random random) {
        ComplexVector randomVector = new ComplexVector(dimension, Mode.HERMITIAN);
        float[] coordinates = randomVector.getCoordinates();
        for (int d = 0; d < coordinates.length; ++d) {
            coordinates[d] = (float)((double)random.nextFloat() - 0.5) / (float)coordinates.length;
        }
        return randomVector;
    }

    @Override
    public double measureOverlap(Vector other) {
        IncompatibleVectorsException.checkVectorsCompatible(this, other);
        if (this.isZeroVector()) {
            return 0.0;
        }
        ComplexVector complexOther = (ComplexVector)other;
        if (complexOther.isZeroVector()) {
            return 0.0;
        }
        switch (DOMINANT_MODE) {
            case HERMITIAN: {
                return this.measureHermitianOverlap(complexOther);
            }
            case CARTESIAN: {
                return this.measureCartesianAngularOverlap(complexOther);
            }
            case POLAR_DENSE: {
                return this.measurePolarDenseOverlap(complexOther);
            }
            case POLAR_SPARSE: {
                throw new IllegalArgumentException("POLAR_SPARSE is not allowed as DOMINANT_MODE.");
            }
        }
        return 0.0;
    }

    protected double measureHermitianOverlap(ComplexVector other) {
        other.toCartesian();
        double result = 0.0;
        double norm1 = 0.0;
        double norm2 = 0.0;
        for (int i = 0; i < this.dimension * 2; ++i) {
            result += (double)(this.coordinates[i] * other.coordinates[i]);
            norm1 += (double)(this.coordinates[i] * this.coordinates[i]);
            norm2 += (double)(other.coordinates[i] * other.coordinates[i]);
        }
        return result / Math.sqrt(norm1 * norm2);
    }

    protected double measureCartesianAngularOverlap(ComplexVector other) {
        this.toCartesian();
        other.toCartesian();
        double cumulativeCosine = 0.0;
        int nonZeroDimensionPairs = 0;
        for (int i = 0; i < this.dimension * 2; i += 2) {
            double resultThisPair = this.coordinates[i] * other.coordinates[i];
            resultThisPair += (double)(this.coordinates[i + 1] * other.coordinates[i + 1]);
            double norm1 = this.coordinates[i] * this.coordinates[i];
            norm1 += (double)(this.coordinates[i + 1] * this.coordinates[i + 1]);
            double norm2 = other.coordinates[i] * other.coordinates[i];
            norm2 += (double)(other.coordinates[i + 1] * other.coordinates[i + 1]);
            norm1 = Math.sqrt(norm1);
            norm2 = Math.sqrt(norm2);
            if (!(norm1 > 0.0) || !(norm2 > 0.0)) continue;
            cumulativeCosine += resultThisPair / (norm1 * norm2);
            ++nonZeroDimensionPairs;
        }
        return nonZeroDimensionPairs != 0 ? cumulativeCosine / (double)nonZeroDimensionPairs : 0.0;
    }

    protected double measurePolarDenseOverlap(ComplexVector other) {
        this.toDensePolar();
        other.toDensePolar();
        int nonZeroEntries = 0;
        short[] phaseAnglesOther = other.getPhaseAngles();
        float sum = 0.0f;
        for (int i = 0; i < this.dimension; i = (int)((short)(i + 1))) {
            if (this.phaseAngles[i] == -1) continue;
            ++nonZeroEntries;
            if (phaseAnglesOther[i] == -1) continue;
            sum += CircleLookupTable.getRealEntry((short)Math.abs(this.phaseAngles[i] - phaseAnglesOther[i]));
        }
        return sum / (float)nonZeroEntries;
    }

    @Override
    public void normalize() {
        if (this.isZeroVector()) {
            return;
        }
        switch (DOMINANT_MODE) {
            case HERMITIAN: {
                this.normalizeHermitian();
                return;
            }
            case CARTESIAN: {
                this.normalizeCartesian();
                return;
            }
            case POLAR_DENSE: {
                this.toDensePolar();
                return;
            }
            case POLAR_SPARSE: {
                throw new IllegalArgumentException("POLAR_SPARSE is not allowed as DOMINANT_MODE.");
            }
        }
    }

    public void normalizeCartesian() {
        this.toDensePolar();
        this.toCartesian();
    }

    protected void normalizeHermitian() {
        int x;
        float[] coords = this.getCoordinates();
        float norm = 0.0f;
        for (x = 0; x < coords.length; ++x) {
            norm = (float)((double)norm + Math.pow(coords[x], 2.0));
        }
        norm = (float)Math.sqrt(norm);
        for (x = 0; x < coords.length; ++x) {
            coords[x] = coords[x] / norm;
        }
    }

    @Override
    public void superpose(Vector other, double weight, int[] permutation) {
        IncompatibleVectorsException.checkVectorsCompatible(this, other);
        ComplexVector complexOther = (ComplexVector)other;
        if (this.opMode != Mode.CARTESIAN) {
            this.toCartesian();
        }
        switch (complexOther.opMode) {
            case CARTESIAN: 
            case HERMITIAN: {
                ComplexVectorUtils.superposeWithCoord(this, complexOther, (float)weight, permutation);
                break;
            }
            case POLAR_SPARSE: {
                ComplexVectorUtils.superposeWithSparseAngle(this, complexOther, (float)weight, permutation);
                break;
            }
            case POLAR_DENSE: {
                ComplexVectorUtils.superposeWithAngle(this, complexOther, (float)weight, permutation);
            }
        }
    }

    public void toCartesian() {
        switch (this.opMode) {
            case HERMITIAN: {
                return;
            }
            case CARTESIAN: {
                return;
            }
            case POLAR_SPARSE: {
                this.sparsePolarToCartesian();
                return;
            }
            case POLAR_DENSE: {
                this.densePolarToCartesian();
            }
        }
    }

    private void sparsePolarToCartesian() {
        assert (this.opMode == Mode.POLAR_SPARSE);
        this.sparsePolarToDensePolar();
        this.densePolarToCartesian();
    }

    private void densePolarToCartesian() {
        assert (this.opMode == Mode.POLAR_DENSE);
        this.coordinates = new float[this.dimension * 2];
        for (int i = 0; i < this.dimension; ++i) {
            this.coordinates[2 * i] = CircleLookupTable.getRealEntry(this.phaseAngles[i]);
            this.coordinates[2 * i + 1] = CircleLookupTable.getImagEntry(this.phaseAngles[i]);
        }
        this.opMode = Mode.CARTESIAN;
        this.phaseAngles = null;
    }

    public void toDensePolar() {
        switch (this.opMode) {
            case POLAR_DENSE: {
                return;
            }
            case POLAR_SPARSE: {
                this.sparsePolarToDensePolar();
                return;
            }
            case CARTESIAN: {
                this.cartesianToDensePolar();
                return;
            }
            case HERMITIAN: {
                this.cartesianToDensePolar();
                return;
            }
        }
    }

    private void cartesianToDensePolar() {
        assert (this.opMode == Mode.CARTESIAN || this.opMode == Mode.HERMITIAN);
        this.opMode = Mode.POLAR_DENSE;
        this.phaseAngles = new short[this.dimension];
        for (int i = 0; i < this.dimension; ++i) {
            this.phaseAngles[i] = CircleLookupTable.phaseAngleFromCartesianTrig(this.coordinates[2 * i], this.coordinates[2 * i + 1]);
        }
        this.coordinates = null;
    }

    private void sparsePolarToDensePolar() {
        int i;
        assert (this.opMode == Mode.POLAR_SPARSE);
        this.phaseAngles = new short[this.dimension];
        for (i = 0; i < this.dimension; ++i) {
            this.phaseAngles[i] = -1;
        }
        if (this.sparseOffsets == null) {
            return;
        }
        for (i = 0; i < this.sparseOffsets.length; i += 2) {
            short positionToAdd = this.sparseOffsets[i];
            int phaseAngleIdx = i + 1;
            this.phaseAngles[positionToAdd] = this.sparseOffsets[phaseAngleIdx];
        }
        this.opMode = Mode.POLAR_DENSE;
        this.sparseOffsets = null;
    }

    @Override
    public void bind(Vector other) {
        IncompatibleVectorsException.checkVectorsCompatible(this, other);
        ComplexVector complexOther = (ComplexVector)other;
        this.convolve(complexOther, 1);
    }

    @Override
    public void release(Vector other) {
        IncompatibleVectorsException.checkVectorsCompatible(this, other);
        ComplexVector complexOther = (ComplexVector)other;
        this.convolve(complexOther, -1);
    }

    public void convolve(ComplexVector other, int direction) {
        IncompatibleVectorsException.checkVectorsCompatible(this, other);
        if (this.getOpMode().equals((Object)Mode.HERMITIAN) && other.getOpMode().equals((Object)Mode.HERMITIAN)) {
            this.convolveCartesian(other, direction);
        } else {
            this.toDensePolar();
            ComplexVector otherCopy = other.copy();
            otherCopy.toDensePolar();
            short[] otherAngles = otherCopy.getPhaseAngles();
            for (int i = 0; i < this.dimension; ++i) {
                if (otherAngles[i] == -1) continue;
                if (this.phaseAngles[i] == -1) {
                    this.phaseAngles[i] = otherAngles[i];
                    continue;
                }
                short angleToAdd = otherAngles[i];
                if (direction <= 0) {
                    angleToAdd = (short)(16384 - angleToAdd);
                }
                this.phaseAngles[i] = (short)((this.phaseAngles[i] + angleToAdd) % 16384);
            }
        }
    }

    public void convolveCartesian(ComplexVector other, int direction) {
        IncompatibleVectorsException.checkVectorsCompatible(this, other);
        float[] norms = new float[this.dimension];
        float[] otherNorms = new float[this.dimension];
        for (int q = 0; q < this.dimension; ++q) {
            float norm = 0.0f;
            float othernorm = 0.0f;
            norm = (float)((double)norm + Math.pow(this.coordinates[q * 2], 2.0));
            norm = (float)((double)norm + Math.pow(this.coordinates[2 * q + 1], 2.0));
            othernorm = (float)((double)othernorm + Math.pow(other.coordinates[q * 2], 2.0));
            othernorm = (float)((double)othernorm + Math.pow(other.coordinates[2 * q + 1], 2.0));
            norm = (float)Math.sqrt(norm);
            othernorm = (float)Math.sqrt(othernorm);
            norms[q] = norm;
            otherNorms[q] = othernorm;
        }
        this.toDensePolar();
        ComplexVector otherCopy = other.copy();
        otherCopy.toDensePolar();
        short[] otherAngles = otherCopy.getPhaseAngles();
        for (int i = 0; i < this.dimension; ++i) {
            if (otherAngles[i] == -1) continue;
            if (this.phaseAngles[i] == -1) {
                this.phaseAngles[i] = otherAngles[i];
                continue;
            }
            short angleToAdd = otherAngles[i];
            if (direction <= 0) {
                angleToAdd = (short)(16384 - angleToAdd);
            }
            this.phaseAngles[i] = (short)((this.phaseAngles[i] + angleToAdd) % 16384);
        }
        this.toCartesian();
        double newNorm = 0.0;
        for (int q = 0; q < this.dimension; ++q) {
            int n = q * 2;
            this.coordinates[n] = this.coordinates[n] * (norms[q] * otherNorms[q]);
            int n2 = q * 2 + 1;
            this.coordinates[n2] = this.coordinates[n2] * (norms[q] * otherNorms[q]);
        }
        this.normalizeHermitian();
    }

    public void complement() {
        assert (this.opMode == Mode.POLAR_DENSE);
        int t = 8192;
        int i = 0;
        while (i < this.dimension) {
            int n = i++;
            this.phaseAngles[n] = (short)(this.phaseAngles[n] + t);
        }
    }

    @Override
    public void writeToLuceneStream(IndexOutput outputStream) {
        this.toCartesian();
        for (int i = 0; i < this.dimension * 2; ++i) {
            try {
                outputStream.writeInt(Float.floatToIntBits(this.coordinates[i]));
                continue;
            }
            catch (ClosedByInterruptException e) {
                throw new QueryInterruptedException("Transaction was aborted by the user");
            }
            catch (IOException e) {
                logger.severe(e.toString());
            }
        }
    }

    @Override
    public void writeToLuceneStream(IndexOutput outputStream, int k) {
        this.toCartesian();
        for (int i = 0; i < k * 2; ++i) {
            try {
                outputStream.writeInt(Float.floatToIntBits(this.coordinates[i]));
                continue;
            }
            catch (ClosedByInterruptException e) {
                throw new QueryInterruptedException("Transaction was aborted by the user");
            }
            catch (IOException e) {
                logger.severe(e.toString());
            }
        }
    }

    @Override
    public void readFromLuceneStream(IndexInput inputStream) {
        this.opMode = Mode.CARTESIAN;
        this.coordinates = new float[this.dimension * 2];
        for (int i = 0; i < this.dimension * 2; ++i) {
            try {
                this.coordinates[i] = Float.intBitsToFloat(inputStream.readInt());
                continue;
            }
            catch (IOException e) {
                logger.severe("Failed to parse vector from Lucene stream.  This signifies a programming or runtime error, e.g., a dimension mismatch.");
                logger.severe(e.toString());
            }
        }
    }

    @Override
    public void readFromByteBuffer(ByteBuffer byteBuffer) {
        this.opMode = Mode.CARTESIAN;
        this.coordinates = new float[this.dimension * 2];
        for (int i = 0; i < this.dimension * 2; ++i) {
            this.coordinates[i] = Float.intBitsToFloat(byteBuffer.getInt());
        }
    }

    @Override
    public String writeToString() {
        this.toCartesian();
        StringBuilder builder = new StringBuilder();
        for (int i = 0; i < this.coordinates.length; ++i) {
            builder.append(Float.toString(this.coordinates[i]));
            if (i == this.coordinates.length - 1) continue;
            builder.append("|");
        }
        return builder.toString();
    }

    @Override
    public void readFromString(String input) {
        this.toCartesian();
        String[] entries = input.split("\\|");
        switch (this.opMode) {
            case CARTESIAN: {
                if (entries.length != this.dimension * 2) {
                    throw new IllegalArgumentException("Found " + entries.length + " possible coordinates: expected " + this.dimension * 2);
                }
                if (this.coordinates.length == 0) {
                    this.coordinates = new float[this.dimension];
                }
                for (int i = 0; i < this.coordinates.length; ++i) {
                    this.coordinates[i] = Float.parseFloat(entries[i]);
                }
                break;
            }
            case POLAR_DENSE: {
                if (entries.length != this.dimension) {
                    throw new IllegalArgumentException("Found " + entries.length + " possible coordinates: expected " + this.dimension);
                }
                if (this.phaseAngles == null || this.phaseAngles.length == 0) {
                    this.phaseAngles = new short[this.dimension];
                }
                for (int i = 0; i < this.phaseAngles.length; ++i) {
                    this.phaseAngles[i] = (short)Integer.parseInt(entries[i]);
                }
                break;
            }
            case POLAR_SPARSE: {
                logger.info("Reading sparse complex vector from string is not supported.");
            }
        }
    }

    protected ComplexVector(float[] coordinates) {
        this.dimension = coordinates.length / 2;
        this.coordinates = coordinates;
        this.opMode = Mode.CARTESIAN;
    }

    protected ComplexVector(short[] phaseAngles) {
        this.dimension = phaseAngles.length;
        this.phaseAngles = phaseAngles;
        this.opMode = Mode.POLAR_DENSE;
    }

    public float[] getCoordinates() {
        return this.coordinates;
    }

    public void setCoordinates(float[] coordinates) {
        this.coordinates = coordinates;
    }

    public short[] getPhaseAngles() {
        return this.phaseAngles;
    }

    protected void setPhaseAngles(short[] phaseAngles) {
        this.phaseAngles = phaseAngles;
    }

    protected short[] getSparseOffsets() {
        return this.sparseOffsets;
    }

    protected void setSparseOffsets(short[] sparseOffsets) {
        this.sparseOffsets = sparseOffsets;
    }

    @Override
    public int getDimension() {
        return this.dimension;
    }

    protected Mode getOpMode() {
        return this.opMode;
    }

    protected void setOpMode(Mode opMode) {
        this.opMode = opMode;
    }

    public static enum Mode {
        POLAR_DENSE,
        POLAR_SPARSE,
        CARTESIAN,
        HERMITIAN;

    }
}

