/*
 * Decompiled with CFR 0.152.
 */
package com.ontotext.graphdb.plugin.history;

import com.google.common.annotations.VisibleForTesting;
import com.ontotext.graphdb.plugin.history.CollectionHistory;
import com.ontotext.graphdb.plugin.history.HistoryConnection;
import com.ontotext.graphdb.plugin.history.HistoryPlugin;
import com.ontotext.graphdb.plugin.history.collection.HistoryCollectionIterator;
import com.ontotext.trree.sdk.PluginException;
import java.io.IOException;
import java.nio.file.FileVisitOption;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;
import java.time.Duration;
import java.time.Instant;
import java.time.Period;
import java.time.ZoneId;
import java.time.ZoneOffset;
import java.time.ZonedDateTime;
import java.time.format.DateTimeParseException;
import java.util.Locale;
import java.util.function.Supplier;
import java.util.stream.Stream;
import org.eclipse.rdf4j.model.Literal;

public class HistoryEditor {
    private static final String INDEX_SUFFIX = ".index";
    private static final String BACKUP_SUFFIX = ".bak";
    private final Path historyFile;
    private final Path historyIndexFile;
    private final Path historyMetadataDirectory;
    private final Path historyBackupFile;
    private final Path historyIndexBackupFile;
    private final Path historyMetadataBackupDirectory;
    private final Path historyTmpFile;
    private final Path historyIndexTmpFile;
    private final Path historyMetadataTmpDirectory;
    private final HistoryPlugin historyPlugin;
    private final Path tmpEditDirectory;
    private final int entityIdSize;
    private boolean appliedHistoryCollection;
    private Supplier<Boolean> editOperation;

    HistoryEditor(HistoryPlugin historyPlugin) {
        this.historyPlugin = historyPlugin;
        Path mainDir = historyPlugin.getDataDir().toPath();
        try {
            this.tmpEditDirectory = Files.createTempDirectory(mainDir, "edit-history", new FileAttribute[0]);
        }
        catch (IOException e) {
            throw new PluginException("Could not create temporary directory for editing history", (Throwable)e);
        }
        this.historyFile = mainDir.resolve("history");
        this.historyIndexFile = mainDir.resolve("history.index");
        this.historyMetadataDirectory = mainDir.resolve("tx-metadata");
        this.historyBackupFile = mainDir.resolve("history.bak");
        this.historyIndexBackupFile = mainDir.resolve("history.index.bak");
        this.historyMetadataBackupDirectory = mainDir.resolve("tx-metadata.bak");
        this.historyTmpFile = this.tmpEditDirectory.resolve("history");
        this.historyIndexTmpFile = this.tmpEditDirectory.resolve("history.index");
        this.historyMetadataTmpDirectory = this.tmpEditDirectory.resolve("tx-metadata");
        this.entityIdSize = historyPlugin.entityIdSize;
    }

    public void clearHistory() {
        this.editOperation = () -> {
            this.historyPlugin.getLogger().info("Clearing the entire history");
            return true;
        };
    }

    public void trimHistoryToPeriod(Literal period) {
        try {
            Instant momentToKeepAfter = this.subtractPeriodFromNow(period.stringValue());
            long momentInMillis = momentToKeepAfter.toEpochMilli();
            this.editOperation = () -> {
                this.historyPlugin.getLogger().info("Trimming history to period {}, keeping all history after {}", (Object)period.stringValue(), (Object)ZonedDateTime.ofInstant(momentToKeepAfter, ZoneId.systemDefault()));
                return this.trimHistoryByTimeLimit(momentInMillis);
            };
        }
        catch (DateTimeParseException e) {
            throw new PluginException("Not a valid xsd:duration period literal: " + period, (Throwable)e);
        }
    }

    public void trimHistoryBefore(Literal datetime) {
        try {
            Instant instant = datetime.calendarValue().toGregorianCalendar().toInstant();
            long millis = instant.toEpochMilli();
            this.editOperation = () -> {
                this.historyPlugin.getLogger().info("Trimming history before {}, keeping all history after it", (Object)ZonedDateTime.ofInstant(instant, ZoneId.systemDefault()));
                return this.trimHistoryByTimeLimit(millis);
            };
        }
        catch (DateTimeParseException e) {
            throw new PluginException("Not a valid xsd:date or xsd:dateTime literal: " + datetime, (Throwable)e);
        }
    }

    private boolean trimHistoryByTimeLimit(long minDate) {
        try (CollectionHistory temporaryHistory = new CollectionHistory(this.tmpEditDirectory, this.entityIdSize);
             HistoryConnection temporaryHistoryConnection = temporaryHistory.getConnection();){
            temporaryHistoryConnection.begin();
            try (HistoryCollectionIterator historyCollectionIterator = this.historyPlugin.historyWriteConnection.getHistoryCollectionIterator(minDate, 0L, 0L, 0L, 0L, 0L);){
                while (historyCollectionIterator.next()) {
                    temporaryHistoryConnection.putFrom(historyCollectionIterator, this.historyPlugin.historyWriteConnection);
                }
            }
            catch (Exception e) {
                temporaryHistoryConnection.rollback();
                throw new PluginException("An error occurred while writing to temporary history file", (Throwable)e);
            }
            temporaryHistoryConnection.precommit();
            temporaryHistoryConnection.commit();
        }
        return true;
    }

    public void trimToSize(Literal size) {
        try {
            long longSize = size.longValue();
            if (longSize < 0L) {
                throw new PluginException("Size for trimToSize must be non-negative: " + longSize);
            }
            this.editOperation = () -> {
                this.historyPlugin.getLogger().info("Trimming history by size, keeping at most {} entries", (Object)String.format("%,d", longSize));
                return this.trimToSize(longSize);
            };
        }
        catch (NumberFormatException e) {
            throw new PluginException("Not a valid long number: " + size, (Throwable)e);
        }
    }

    private boolean trimToSize(long numberOfStatementsToRemain) {
        try (CollectionHistory temporaryHistory = new CollectionHistory(this.tmpEditDirectory, this.entityIdSize);
             HistoryConnection temporaryHistoryConnection = temporaryHistory.getConnection();){
            temporaryHistoryConnection.begin();
            try (HistoryCollectionIterator historyCollectionIterator = this.historyPlugin.historyWriteConnection.getHistoryCollectionIterator(0L, 0L, 0L, 0L, 0L, 0L);){
                while (historyCollectionIterator.next() && numberOfStatementsToRemain-- > 0L) {
                    temporaryHistoryConnection.putFrom(historyCollectionIterator, this.historyPlugin.historyWriteConnection);
                }
            }
            catch (Exception e) {
                temporaryHistoryConnection.rollback();
                throw new PluginException("An error occurred while writing to temporary history file", (Throwable)e);
            }
            temporaryHistoryConnection.precommit();
            temporaryHistoryConnection.commit();
        }
        return true;
    }

    public void precommit() {
        if (this.editOperation != null && this.editOperation.get().booleanValue()) {
            this.historyPlugin.closeHistory();
            this.applyNewHistory();
            this.historyPlugin.openHistory();
            this.appliedHistoryCollection = true;
        }
    }

    public boolean commit() {
        if (this.appliedHistoryCollection) {
            this.deleteBackupHistory();
            this.deleteTemporaryHistory();
            return true;
        }
        return false;
    }

    public void rollback() {
        if (this.appliedHistoryCollection) {
            this.historyPlugin.closeHistory();
            this.restoreBackupHistory();
            this.historyPlugin.openHistory();
        }
        this.deleteTemporaryHistory();
    }

    private void applyNewHistory() {
        this.renameFile(this.historyFile, this.historyBackupFile);
        this.renameFile(this.historyTmpFile, this.historyFile);
        this.renameFile(this.historyIndexFile, this.historyIndexBackupFile);
        this.renameFile(this.historyIndexTmpFile, this.historyIndexFile);
        this.renameFile(this.historyMetadataDirectory, this.historyMetadataBackupDirectory);
        this.renameFile(this.historyMetadataTmpDirectory, this.historyMetadataDirectory);
    }

    private void restoreBackupHistory() {
        this.deleteTemporaryHistory();
        this.renameFile(this.historyBackupFile, this.historyFile);
        this.renameFile(this.historyIndexBackupFile, this.historyIndexFile);
        this.revertDirectory(this.historyMetadataBackupDirectory, this.historyMetadataDirectory);
    }

    private void renameFile(Path oldFile, Path newFile) {
        if (Files.exists(oldFile, new LinkOption[0])) {
            try {
                if (Files.isDirectory(newFile, new LinkOption[0])) {
                    this.deleteDirectory(newFile);
                }
                Files.move(oldFile, newFile, StandardCopyOption.REPLACE_EXISTING);
            }
            catch (IOException e) {
                throw new PluginException("Could not rename file '" + oldFile + "' to '" + newFile + "'", (Throwable)e);
            }
        }
    }

    private void revertDirectory(final Path oldFile, final Path newFile) {
        if (Files.exists(oldFile, new LinkOption[0]) && Files.isDirectory(oldFile, new LinkOption[0])) {
            try {
                Files.walkFileTree(oldFile, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

                    @Override
                    public FileVisitResult preVisitDirectory(Path dir, BasicFileAttributes attrs) throws IOException {
                        if (!Files.exists(newFile, new LinkOption[0])) {
                            Files.createDirectories(newFile, new FileAttribute[0]);
                        }
                        return FileVisitResult.CONTINUE;
                    }

                    @Override
                    public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
                        Files.move(file, newFile.resolve(oldFile.relativize(file)), StandardCopyOption.REPLACE_EXISTING);
                        return FileVisitResult.CONTINUE;
                    }

                    @Override
                    public FileVisitResult postVisitDirectory(Path dir, IOException exc) throws IOException {
                        Files.delete(dir);
                        return FileVisitResult.CONTINUE;
                    }
                });
            }
            catch (IOException e) {
                throw new PluginException("Could not rename directory '" + oldFile + "' to '" + newFile + "'", (Throwable)e);
            }
        }
    }

    public void deleteBackupHistory() {
        this.deleteFile(this.historyBackupFile);
        this.deleteFile(this.historyIndexBackupFile);
        this.deleteDirectory(this.historyMetadataBackupDirectory);
    }

    private void deleteTemporaryHistory() {
        this.deleteDirectory(this.tmpEditDirectory);
    }

    private void deleteFile(Path file) {
        try {
            Files.deleteIfExists(file);
        }
        catch (IOException e) {
            this.historyPlugin.getLogger().warn("Could not delete file: " + file, (Throwable)e);
        }
    }

    private void deleteDirectory(Path directory) {
        if (Files.exists(directory, new LinkOption[0])) {
            try (Stream<Path> walkStream = Files.walk(directory, new FileVisitOption[0]);){
                walkStream.sorted((path1, path2) -> -path1.compareTo((Path)path2)).forEach(path -> {
                    try {
                        Files.deleteIfExists(path);
                    }
                    catch (IOException e) {
                        this.historyPlugin.getLogger().warn("Error deleting file or directory: {}", path);
                    }
                });
            }
            catch (IOException e) {
                this.historyPlugin.getLogger().warn("Could not delete directory: {}", (Object)directory, (Object)e);
            }
        }
    }

    private Instant subtractPeriodFromNow(String periodString) {
        return HistoryEditor.subtractPeriod(ZonedDateTime.now(ZoneOffset.UTC), periodString);
    }

    @VisibleForTesting
    static Instant subtractPeriod(ZonedDateTime moment, String periodString) {
        if (periodString.startsWith("-")) {
            throw new PluginException("The period must be non-negative: " + periodString);
        }
        String[] periodComponents = periodString.toUpperCase(Locale.ROOT).split("T", 2);
        if (periodComponents[0].length() > 1) {
            Period p = Period.parse(periodComponents[0]);
            moment = moment.minus(p);
        }
        if (periodComponents.length == 2) {
            Duration d = Duration.parse("PT" + periodComponents[1]);
            moment = moment.minus(d);
        }
        return moment.toInstant();
    }
}

