/**
 * Top level function for value generation. It's only argument should be a list of JsValue
 * instances that should be evaluated.
 *
 * @param templates the list of templates to process
 */
function generate(templates) {
    templates.forEach(template => evalValue(template));
}

/**
 * Evaluates the given computed JsValue template using the JS eval() function and sets back it's
 * value when finished. If the value is already evaluated then calling this method will have no
 * effect. If the given value is not lazy then nothing will be done as well.
 *
 * @param template the JsValue to process.
 */
function evalValue(template) {
    if (!template.isLazy()) {
        // not lazy value, can't continue as we will not find the used methods
        return;
    }
    if (!template.isEmpty()) {
        // the value is already set so do nothing
        return;
    }
    let resolver = template.getValueResolver();
    let value = eval(resolver.getEvaluation());
    if (value && isValueNotResolved(value)) {
        // if the value is not evaluated for some reason or results to invalid java type then
        // set as null to prevent mutation side effects
        resolver.setComputedValue(null);
    } else {
        resolver.setComputedValue(value);
    }
}

/**
 * Template function that process any non IRI generation templates and executes the logic needed
 * for recursive property referencing. This means if a property references other computed property
 * and that property is not already evaluated it will be evaluated before applying it to the result.
 *
 * @param strings the list of string literals from the template. They are always 1 more than the
 *      expressions
 * @param values the list of evaluated expressions from the template.
 * @returns returns a string representation of the evaluated template
 */
function genValue(strings, ...values) {
    if (values.length === 0) {
        // template without expressions
        return strings[0];
    }
    let result = "";
    strings.forEach(function (prefix, index) {
        if (index >= values.length) {
            result += prefix
        } else {
            let value = values[index];
            if (typeof value === 'object') {
                // try to eval non evaluated values
                if (value.isLazy()) {
                    evalValue(value);
                }
                if (isValueNotResolved(value)) {
                    // we do not have this value so skip it
                    value = "";
                }
            }
            result += prefix + value;
        }
    });
    return result;
}

/**
 * Template function that process all IRI generation templates and executes the logic needed
 * for recursive property referencing and IRI escaping. This means if a property references other
 * computed property and that property is not already evaluated it will be evaluated before applying
 * it to the result. The method also ensures the escape methods are called on the referenced
 * properties if not already called.
 *
 * @param strings the list of string literals from the template. They are always 1 more than the
 *      expressions
 * @param values the list of evaluated expressions from the template.
 * @returns returns a string representation of the evaluated template
 */
function _iri(strings, ...values) {
    let result = "";
    strings.forEach(function (prefix, index) {
        if (index === 0) {
            prefix = resolveIriPrefix(prefix);
        }
        if (index >= values.length) {
            result += prefix;
        } else {
            let value = values[index];
            if (typeof value === 'object') {
                // in case we have an expression we should make sure to escape it if not already
                // this works as the value itself is JsValue instance and the escaping is applied
                // on the toString method. Also if the escape is already been called this will
                // do nothing to the end result
                value.escape();
                if (value.isIri()) {
                    value.compact();
                }
                // try to eval non evaluated values
                if (value.isLazy()) {
                    evalValue(value);
                }
                if (isValueNotResolved(value)) {
                    // we do not have value so skip it and it's leading separator
                    value = "";
                    prefix = prefix.replace(/[\W_]+$/, "");
                }
            }
            result += prefix + value;
        }
    });
    return result;
}

/**
 * For the given template token tries to resolve if the token should be expanded with a
 * prefix IRI or base IRI or not modified at all.
 *
 * @param item the token value to check
 * @returns updated token value
 * @throws an error if undefined prefix is used in the token
 */
function resolveIriPrefix(item) {
    // we do not need the leading separator if any
    let trimmedValue = item.replace(/^\W+/, "");
    if (trimmedValue.includes(":")) {
        let prefix = trimmedValue.substring(0, trimmedValue.indexOf(":"));
        if (isLeadingSchema(prefix)) {
            return trimmedValue;
        } else if (_prefix[prefix]) {
            return _prefix[prefix] + trimmedValue.substr(trimmedValue.indexOf(":") + 1);
        } else {
            throw `Invalid iri pattern prefix '${trimmedValue}'`;
        }
    }
    return _prefix.base_iri + trimmedValue;
}

/**
 * Checks if the given token begins with one of the default URL schemas.
 * Currently the checked values are: http, https, mailto, urn, geo
 *
 * @param value the value to check.
 * @returns true if the token begins with one of the supported URL schemas
 */
function isLeadingSchema(value) {
    return value.startsWith('http')
        || value.startsWith('https')
        || value.startsWith('mailto')
        || value.startsWith('urn')
        || value.startsWith('geo');
}

/**
 * Checks if the given JsValue is resolved property or not.<br>
 * For string based values this means the value is not 'null' or ''.<br>
 * For number based values it should not be 'NaN' or 'Infinity'.<br>
 *
 * @param value the JsValue instance to check
 * @returns  if the value is invalid
 */
function isValueNotResolved(value) {
    const val = value.toString();
    return !val
        || val === 'null'
        || val === 'NaN'
        || val === 'Infinity'
        || val === '';
}
