export declare class ObjectUtil {
    /**
     * Performs a deep equality check between two values.
     *
     * This function compares two values to determine if they are deeply equal.
     * It supports comparison of primitives, arrays, and objects, including nested structures.
     *
     * @param obj1 - The first value to compare.
     * @param obj2 - The second value to compare.
     * @returns `true` if both values are deeply equal, otherwise `false`.
     *
     * @example
     * // Comparing two deeply nested objects
     * const obj1 = { a: 1, b: { c: [1, 2, 3], d: { e: 'hello' } } };
     * const obj2 = { a: 1, b: { c: [1, 2, 3], d: { e: 'hello' } } };
     * console.log(deepEqual(obj1, obj2)); // true
     *
     * @example
     * // Comparing two arrays with different values
     * const arr1 = [1, 2, 3];
     * const arr2 = [1, 2, 4];
     * console.log(deepEqual(arr1, arr2)); // false
     */
    static deepEqual(obj1: unknown, obj2: unknown): boolean;
    /**
     * Creates a deep copy of the given object.
     *
     * This method recursively copies all properties and values of the object,
     * handling different types such as primitives, arrays, and plain objects.
     * It ensures that the copied object does not share references with the original object,
     * meaning that all nested structures are also copied independently.
     *
     * @param obj - The object to be deeply copied. Can be any type, including primitives,
     *              arrays, or objects.
     * @returns A deep copy of the object. If the input is a primitive, null, undefined,
     *          or a function, the original value is returned as-is.
     */
    static deepCopy(obj: unknown): unknown;
    /**
     * Type guard to check if the value has a 'copy' method.
     * @param value - The value to check.
     * @returns True if the value has a 'copy' method, false otherwise.
     */
    static hasCopyMethod(value: unknown): boolean;
}
