/*
 * Decompiled with CFR 0.152.
 */
package com.ontotext.graphdb.raft.recovery;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.ontotext.graphdb.Config;
import com.ontotext.graphdb.raft.ClusterGroup;
import com.ontotext.graphdb.raft.RecoveryState;
import com.ontotext.graphdb.raft.grpc.NodeException;
import com.ontotext.graphdb.raft.grpc.ReportingIterator;
import com.ontotext.graphdb.raft.grpc.RpcNodeClient;
import com.ontotext.graphdb.raft.grpc.SnapshotData;
import com.ontotext.graphdb.raft.grpc.SnapshotResponse;
import com.ontotext.graphdb.raft.recovery.RaftRecoveryException;
import com.ontotext.graphdb.raft.recovery.RecoveryHook;
import com.ontotext.graphdb.raft.util.RaftUtil;
import io.grpc.Status;
import io.grpc.StatusRuntimeException;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.io.RandomAccessFile;
import java.nio.channels.FileChannel;
import java.nio.file.Paths;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class RecoveryTask
implements Runnable {
    private static final Logger logger = LoggerFactory.getLogger(RecoveryTask.class);
    protected final ClusterGroup clusterGroup;
    protected final RecoveryHook hook;
    protected final AtomicReference<RecoveryState> recoveryState;
    private final Set<String> failedClientAddresses;
    private final ObjectMapper objectMapper = new ObjectMapper();

    public RecoveryTask(ClusterGroup clusterGroup, RecoveryHook hook, AtomicReference<RecoveryState> state) {
        this.clusterGroup = clusterGroup;
        this.hook = hook;
        this.recoveryState = state;
        this.failedClientAddresses = new HashSet<String>(clusterGroup.size() - 1);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void run() {
        while (!Thread.currentThread().isInterrupted()) {
            boolean locked = false;
            boolean threwError = false;
            try {
                locked = this.clusterGroup.lockSnapshotApply();
                this.recoverFromSnapshot();
                return;
            }
            catch (Exception e) {
                logger.error("Error occurred during attempted recovery: ", (Throwable)e);
                threwError = true;
            }
            finally {
                try {
                    if (locked) {
                        this.clusterGroup.unlockSnapshotApply();
                        continue;
                    }
                    logger.warn("Could not lock node with current recovery task");
                }
                finally {
                    if (threwError || !locked) continue;
                    this.hook.apply();
                }
            }
        }
        logger.error("Recovery task is interrupted");
    }

    protected void recoverFromSnapshot() {
        File snapshot = null;
        int nodesToRetry = this.clusterGroup.size() - 1;
        int recoveryTimeOut = this.clusterGroup.getRecoveryTimeout();
        while (snapshot == null) {
            try {
                snapshot = this.saveSnapshot();
            }
            catch (Exception e) {
                StatusRuntimeException cause;
                if (e.getCause() instanceof StatusRuntimeException && (cause = (StatusRuntimeException)e.getCause()).getStatus().getCode() == Status.Code.FAILED_PRECONDITION) {
                    this.failedClientAddresses.addAll(this.recoveryState.get().getAffectedNodeAddresses());
                    if (--nodesToRetry == 0) {
                        this.recoveryState.get().setStateRecoveryOperationFailureWarning(this.clusterGroup.getCurrentAddress(), this.failedClientAddresses);
                        logger.error("Critical error occurred during snapshot request: {}", (Object)this.recoveryState.get().getCurrentStateMessage(), (Object)e);
                        nodesToRetry = this.clusterGroup.size() - 1;
                        this.failedClientAddresses.clear();
                        recoveryTimeOut = (int)TimeUnit.MINUTES.toMillis(15L);
                    }
                }
                logger.error("Unable to retrieve snapshot for recovery. Retrying in {} seconds", (Object)(recoveryTimeOut / 1000), (Object)e);
                this.timeoutBeforeRetry(recoveryTimeOut);
                recoveryTimeOut = this.clusterGroup.getRecoveryTimeout();
                this.recoveryState.get().setStateToNoRecovery();
            }
        }
        try {
            long start = System.currentTimeMillis();
            this.recoveryState.get().setStateToApplySnapshot();
            this.applySnapshot(this.clusterGroup, snapshot);
            logger.info("Applied snapshot successfully in {} seconds", (Object)((System.currentTimeMillis() - start) / 1000L));
        }
        catch (Exception e) {
            logger.error("Critical error occurred during snapshot application due to:", (Throwable)e);
            throw e;
        }
        finally {
            this.recoveryState.get().setStateToNoRecovery();
        }
    }

    protected File saveSnapshot() {
        long matchIndex = this.clusterGroup.getTransactionLog().getLastValidLog();
        logger.info("Trying to find node with log match index {}", (Object)matchIndex);
        this.recoveryState.get().setStateToSearchingForNode();
        RpcNodeClient client = this.fetchClient(matchIndex);
        if (client == null) {
            logger.error("Node {} unable to find suitable node for snapshot recovery", (Object)this.clusterGroup.getCurrentAddress());
            throw new RaftRecoveryException("Unable to find suitable node for snapshot recovery");
        }
        logger.info("Requesting snapshot from node {}", (Object)client.getAddress());
        this.recoveryState.get().setStateToWaitingForSnapshot(client.getAddress());
        return this.recordSnapshot(client, matchIndex);
    }

    protected void applySnapshot(ClusterGroup group, File snapshot) {
        try {
            group.getTransactionLog().shutdown(false);
            group.getStateMachine().applySnapshot(snapshot);
            group.getTransactionLog().initialize();
        }
        catch (Exception e) {
            logger.error("[Node {}] Error during snapshot {} application", new Object[]{group.getCurrentAddress(), snapshot, e});
            throw e;
        }
    }

    protected File recordSnapshot(RpcNodeClient client, long matchIndex) {
        ReportingIterator<SnapshotData> iter = client.requestSnapshot(matchIndex);
        String snapshotDir = "";
        try {
            SnapshotData metadata = iter.next();
            SnapshotData.NullableSnapshotMeta.KindCase kindCase = metadata.getMeta().getKindCase();
            if (kindCase != SnapshotData.NullableSnapshotMeta.KindCase.METADATA) {
                throw new RaftRecoveryException(client.getAddress() + " unable to apply snapshot for recovery");
            }
            File oldSnapshot = this.clusterGroup.getStateMachine().getLatestSnapshot();
            if (oldSnapshot != null && oldSnapshot.exists()) {
                snapshotDir = this.clusterGroup.getStateMachine().getLatestSnapshot().getParent();
                try {
                    this.createSnapshotLockFile(snapshotDir, oldSnapshot.getName());
                    logger.info("Deleting older cluster snapshot {}", (Object)oldSnapshot);
                    oldSnapshot.delete();
                    this.releaseSnapshotLockFile(snapshotDir);
                }
                catch (IOException e) {
                    throw new RaftRecoveryException("Error occurred while managing the snapshot lock file for node " + client.getAddress(), e);
                }
            }
            SnapshotData.SnapshotMeta snapshotMeta = metadata.getMeta().getMetadata();
            File snapshot = this.generateSnapshotFile(snapshotMeta);
            RaftUtil.validateEnoughDiskSpace(snapshotMeta.getSize(), snapshot);
            logger.info("Recording snapshot from node {} with last index {}", (Object)client.getAddress(), (Object)snapshotMeta.getMatchIndex());
            this.recoveryState.get().setStateToRecordingSnapshot(client.getAddress());
            try (RandomAccessFile file = new RandomAccessFile(snapshot, "rw");){
                FileChannel fileChannel = file.getChannel();
                fileChannel.write(metadata.getData().asReadOnlyByteBuffer());
                while (iter.hasNext()) {
                    fileChannel.write(iter.next().getData().asReadOnlyByteBuffer());
                }
                logger.info("Finished recording snapshot with last index {}", (Object)snapshotMeta.getMatchIndex());
            }
            catch (IOException e) {
                throw new RaftRecoveryException("Unable to record incoming snapshot from " + client.getAddress(), e);
            }
            return snapshot;
        }
        catch (Exception e) {
            if (!(e instanceof NodeException)) {
                iter.reportError("Error occurred on requesting node", e);
            }
            throw e;
        }
    }

    protected RpcNodeClient fetchClient(long matchIndex) {
        RpcNodeClient matchingIndexClient = null;
        RpcNodeClient leaderClient = null;
        long prevNodeNextLogIndex = 0L;
        for (RpcNodeClient client : this.clusterGroup) {
            if (this.failedClientAddresses.contains(client.getAddress())) continue;
            try {
                SnapshotResponse response = client.sendSnapshotQuery(matchIndex);
                if (response.getStatus() == SnapshotResponse.Status.HAS_SNAPSHOT || response.getStatus() == SnapshotResponse.Status.BUILDING_SNAPSHOT) {
                    if (matchIndex == 0L) {
                        if (response.getLastLogIndex() < prevNodeNextLogIndex) continue;
                        matchingIndexClient = client;
                        prevNodeNextLogIndex = response.getLastLogIndex();
                        continue;
                    }
                    return client;
                }
                if (response.getLastLogIndex() >= prevNodeNextLogIndex && response.getStatus() == SnapshotResponse.Status.FOLLOWER_MATCHES_INDEX) {
                    matchingIndexClient = client;
                    prevNodeNextLogIndex = response.getLastLogIndex();
                    continue;
                }
                if (response.getStatus() != SnapshotResponse.Status.LEADER_MATCHES_INDEX) continue;
                if (response.getLastLogIndex() > prevNodeNextLogIndex) {
                    matchingIndexClient = null;
                }
                leaderClient = client;
                prevNodeNextLogIndex = response.getLastLogIndex();
            }
            catch (Exception e) {
                logger.error("Node {} not available to recover from due to:", (Object)client.getAddress(), (Object)e);
            }
        }
        return matchingIndexClient != null ? matchingIndexClient : leaderClient;
    }

    protected File generateSnapshotFile(SnapshotData.SnapshotMeta metadata) {
        String dir = Paths.get(Config.getDataDirectory(), "raft", "cluster-recovery").toString();
        File file = new File(dir, metadata.getMatchIndex() + metadata.getFormat());
        file.getParentFile().mkdirs();
        try {
            file.createNewFile();
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
        return file;
    }

    private void timeoutBeforeRetry(int recoveryTimeout) {
        try {
            Thread.sleep(recoveryTimeout);
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            logger.error("Unable to recover from out-of-sync", (Throwable)e);
            throw new RaftRecoveryException(e);
        }
    }

    private void createSnapshotLockFile(String snapshotDir, String snapshotToDelete) throws IOException {
        File lockFile = new File(snapshotDir, "snapshot-delete.lock");
        if (lockFile.exists() && lockFile.isFile()) {
            lockFile.delete();
        }
        try (FileOutputStream fos = new FileOutputStream(lockFile);){
            this.objectMapper.writeValue((OutputStream)fos, Map.of("snapshotToDelete", snapshotToDelete));
        }
    }

    private void releaseSnapshotLockFile(String snapshotDir) {
        File lockFile = new File(snapshotDir, "snapshot-delete.lock");
        if (lockFile.exists()) {
            lockFile.delete();
        }
    }
}

