/*
 * Decompiled with CFR 0.152.
 */
package com.ontotext.graphdb.raft.recovery;

import com.azure.storage.blob.specialized.BlobOutputStream;
import com.google.protobuf.ByteString;
import com.google.protobuf.NullValue;
import com.ontotext.graphdb.Config;
import com.ontotext.graphdb.cloud.CloudMultipartUploader;
import com.ontotext.graphdb.cloud.CloudOutputStream;
import com.ontotext.graphdb.cloud.aws.S3BucketOptions;
import com.ontotext.graphdb.cloud.aws.S3MultipartUploader;
import com.ontotext.graphdb.cloud.aws.S3Utils;
import com.ontotext.graphdb.cloud.azure.AzureBlobOptions;
import com.ontotext.graphdb.cloud.azure.AzureBlobUtils;
import com.ontotext.graphdb.cloud.google.GCMultipartUploader;
import com.ontotext.graphdb.cloud.google.GoogleBucketOptions;
import com.ontotext.graphdb.raft.ClusterGroup;
import com.ontotext.graphdb.raft.NodeState;
import com.ontotext.graphdb.raft.StateMachine;
import com.ontotext.graphdb.raft.grpc.AzureBackupRequest;
import com.ontotext.graphdb.raft.grpc.AzureBackupResponse;
import com.ontotext.graphdb.raft.grpc.BackupRequest;
import com.ontotext.graphdb.raft.grpc.GoogleBackupRequest;
import com.ontotext.graphdb.raft.grpc.GoogleBackupResponse;
import com.ontotext.graphdb.raft.grpc.NodeInfo;
import com.ontotext.graphdb.raft.grpc.RecoveryServiceGrpc;
import com.ontotext.graphdb.raft.grpc.S3BackupRequest;
import com.ontotext.graphdb.raft.grpc.S3BackupResponse;
import com.ontotext.graphdb.raft.grpc.SnapshotData;
import com.ontotext.graphdb.raft.grpc.SnapshotRequest;
import com.ontotext.graphdb.raft.grpc.SnapshotResponse;
import com.ontotext.graphdb.raft.node.RaftNode;
import com.ontotext.graphdb.raft.node.concurrent.SemaphoreLock;
import com.ontotext.graphdb.raft.recovery.RaftRecoveryException;
import com.ontotext.graphdb.raft.util.StreamingResponseObserver;
import common.exceptions.InsufficientDiskSpaceException;
import io.grpc.Status;
import io.grpc.StatusRuntimeException;
import io.grpc.stub.StreamObserver;
import java.io.BufferedInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.locks.Lock;
import java.util.function.Supplier;
import org.jetbrains.annotations.NotNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.services.s3.S3AsyncClient;

public class RecoveryService
extends RecoveryServiceGrpc.RecoveryServiceImplBase {
    protected static final long TRY_LOCK_TIMEOUT_MS = 1000L;
    private static final Logger logger = LoggerFactory.getLogger(RecoveryService.class);
    private static final String CLOUD_BACKUP = "cloud backup";
    private static final String BACKUP = "backup";
    private static final String SNAPSHOT = "snapshot";
    private static final String SNAPSHOT_STATUS = "snapshot status";
    private static final int RECOVERY_ATTEMPTS = Config.getPropertyAsInt((String)"graphdb.cluster.node.recovery.attempts", (int)2);
    private final Supplier<RaftNode> supplier;
    private final Lock requestLock;
    private long buildingSnapshotIndex;
    private final Map<String, RecoveryHistory> recoveryResponses = new HashMap<String, RecoveryHistory>();

    public RecoveryService(Supplier<RaftNode> supplier) {
        this.supplier = supplier;
        this.requestLock = new SemaphoreLock();
        this.buildingSnapshotIndex = -1L;
    }

    public void shutdown() {
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void requestSnapshot(SnapshotRequest request, StreamObserver<SnapshotData> responseObserver) {
        RaftNode node = this.fetchAndValidateNode(SNAPSHOT);
        StreamObserver<SnapshotData> delegateObserver = this.fetchBackupObserver(responseObserver);
        try {
            logger.info("Received snapshot build request from {} for match index {}", (Object)request.getNodeId(), (Object)request.getMatchIndex());
            this.validateNode(node, SNAPSHOT);
            this.validateRequest(request, node);
            ClusterGroup clusterGroup = node.getClusterGroup();
            long[] snapshotIndex = new long[1];
            File snapshot = this.fetchSnapshot(request, node, snapshotIndex, false);
            if (this.shouldForceNewSnapshot(request.getNodeId(), snapshotIndex) && snapshotIndex[0] != clusterGroup.getTransactionLog().getLastLogIndex()) {
                logger.info("Forced to build new snapshot for node {} and match index {}", (Object)request.getNodeId(), (Object)request.getMatchIndex());
                snapshot = this.fetchSnapshot(request, node, snapshotIndex, true);
            }
            if (snapshot != null) {
                node.getRecoveryState().get().setStateToSendingSnapshot(request.getNodeId());
                this.sendSnapshot(snapshot, delegateObserver, clusterGroup, snapshotIndex[0]);
                logger.info("Successfully sent snapshot covering index {} to node {}", (Object)snapshotIndex, (Object)request.getNodeId());
            } else {
                delegateObserver.onNext((Object)SnapshotData.newBuilder().setSuccess(false).build());
                delegateObserver.onCompleted();
            }
        }
        catch (Exception e) {
            this.logAndThrowException(SNAPSHOT.toLowerCase(), e, delegateObserver);
        }
        finally {
            node.getRecoveryState().get().setStateToNoRecovery();
        }
    }

    private boolean shouldForceNewSnapshot(String nodeAddress, long[] snapshotIndex) {
        return this.recoveryResponses.computeIfAbsent(nodeAddress, ignored -> new RecoveryHistory()).recordAccess(snapshotIndex[0]);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void requestBackup(BackupRequest request, StreamObserver<SnapshotData> responseObserver) {
        StreamObserver<SnapshotData> delegateObserver = this.fetchBackupObserver(responseObserver);
        try {
            logger.info("Received backup request from {}", (Object)request.getNodeId());
            RaftNode node = this.fetchAndValidateNode(BACKUP);
            this.lockNode(node, BACKUP);
            try {
                this.validateNode(node, BACKUP);
                this.propagateBackup(node, request, delegateObserver);
                delegateObserver.onCompleted();
            }
            finally {
                this.unlockNode(node);
            }
        }
        catch (Exception e) {
            this.logAndThrowException(BACKUP, e, delegateObserver);
        }
    }

    private StreamObserver<SnapshotData> fetchBackupObserver(StreamObserver<SnapshotData> responseObserver) {
        return new StreamingResponseObserver<SnapshotData>(responseObserver);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void hasSnapshot(SnapshotRequest request, StreamObserver<SnapshotResponse> responseObserver) {
        this.requestLock.lock();
        try {
            RaftNode node = this.fetchAndValidateNode(SNAPSHOT);
            if (logger.isDebugEnabled()) {
                logger.debug("Received snapshot query from {} for match index {} with current last log index {}", new Object[]{request.getNodeId(), request.getMatchIndex(), node.getClusterGroup().getTransactionLog().getLastLogIndex()});
            }
            ClusterGroup clusterGroup = node.getClusterGroup();
            long lastLogIndex = node.getClusterGroup().getTransactionLog().getLastLogIndex();
            if (clusterGroup.getStateMachine().getLatestSnapshotIndex() >= request.getMatchIndex() && (request.getMatchIndex() != 0L || clusterGroup.getStateMachine().getLatestSnapshotIndex() == lastLogIndex)) {
                responseObserver.onNext((Object)this.generateResponse(SnapshotResponse.Status.HAS_SNAPSHOT, clusterGroup.getStateMachine().getLatestSnapshotIndex()));
            } else if (clusterGroup.getTransactionLog().getLastLogIndex() >= request.getMatchIndex()) {
                if (this.buildingSnapshotIndex >= request.getMatchIndex()) {
                    responseObserver.onNext((Object)this.generateResponse(SnapshotResponse.Status.BUILDING_SNAPSHOT, this.buildingSnapshotIndex));
                } else if (node.isLeader()) {
                    responseObserver.onNext((Object)this.generateResponse(SnapshotResponse.Status.LEADER_MATCHES_INDEX, lastLogIndex));
                } else {
                    responseObserver.onNext((Object)this.generateResponse(SnapshotResponse.Status.FOLLOWER_MATCHES_INDEX, lastLogIndex));
                }
            } else {
                responseObserver.onNext((Object)this.generateResponse(SnapshotResponse.Status.LOWER_INDEX, lastLogIndex));
            }
            responseObserver.onCompleted();
        }
        catch (Exception e) {
            this.logAndThrowException(SNAPSHOT_STATUS, e, responseObserver);
        }
        finally {
            this.requestLock.unlock();
        }
    }

    @NotNull
    protected RaftNode fetchAndValidateNode(String loggingMethodUser) {
        RaftNode node = this.supplier.get();
        this.validateNode(node, loggingMethodUser);
        return node;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    protected File fetchSnapshot(SnapshotRequest request, RaftNode node, long[] snapshotIndex, boolean forceNew) {
        File snapshot;
        ClusterGroup clusterGroup = node.getClusterGroup();
        if (!forceNew && clusterGroup.getStateMachine().getLatestSnapshotIndex() >= request.getMatchIndex() && request.getMatchIndex() != 0L) {
            logger.info("[Node {}] Already have snapshot covering log index {}", (Object)clusterGroup.getCurrentAddress(), (Object)request.getMatchIndex());
            snapshot = clusterGroup.getStateMachine().getLatestSnapshot();
            snapshotIndex[0] = clusterGroup.getStateMachine().getLatestSnapshotIndex();
        } else {
            this.requestLock.lock();
            try {
                if (request.getMatchIndex() <= clusterGroup.getTransactionLog().getLastLogIndex() && (this.buildingSnapshotIndex < request.getMatchIndex() || request.getMatchIndex() == 0L)) {
                    this.buildingSnapshotIndex = clusterGroup.getTransactionLog().getLastLogIndex();
                }
            }
            finally {
                this.requestLock.unlock();
            }
            node.getRecoveryState().get().setStateToBuildingSnapshot(request.getNodeId());
            snapshot = this.buildSnapshot(node, snapshotIndex, forceNew);
        }
        return snapshot;
    }

    protected void sendSnapshot(File snapshot, StreamObserver<SnapshotData> responseObserver, ClusterGroup group, long snapshotIndex) {
        try (BufferedInputStream stream = new BufferedInputStream(new FileInputStream(snapshot));){
            int read;
            byte[] buffer = new byte[group.getMessageSizeBytes()];
            SnapshotData.NullableSnapshotMeta.Builder noMetadata = SnapshotData.NullableSnapshotMeta.newBuilder().setNull(NullValue.NULL_VALUE);
            SnapshotData.NullableSnapshotMeta.Builder snapshotMeta = SnapshotData.NullableSnapshotMeta.newBuilder().setMetadata(SnapshotData.SnapshotMeta.newBuilder().setFormat(".tar").setMatchIndex(snapshotIndex).setSize(snapshot.length()).setApplyRepositoriesData(true).setApplySystemData(true).setApplyClusterData(true).putAllGroupIds(this.buildGroupMapping(group)).putAllChannelMapping(this.buildRepositoryMapping(group)));
            SnapshotData.Builder builder = SnapshotData.newBuilder().setSuccess(false);
            responseObserver.onNext((Object)builder.setMeta(snapshotMeta).build());
            builder.setMeta(noMetadata);
            while ((read = stream.read(buffer)) > 0) {
                builder.setData(ByteString.copyFrom((byte[])buffer, (int)0, (int)read));
                responseObserver.onNext((Object)builder.build());
            }
            responseObserver.onNext((Object)builder.setData(ByteString.empty()).setSuccess(true).build());
            responseObserver.onCompleted();
        }
        catch (IOException e) {
            throw new RaftRecoveryException(e);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    protected File buildSnapshot(RaftNode node, long[] snapshotIndex, boolean forceNew) {
        node.setBuildingSnapshot(true);
        try {
            block9: {
                while (!node.getClusterGroup().tryLockSnapshotBuild(1000L)) {
                    this.validateNode(node, SNAPSHOT);
                }
                try {
                    this.validateNode(node, SNAPSHOT);
                    long latestSnapshotIndex = node.getClusterGroup().getStateMachine().getLatestSnapshotIndex();
                    if (forceNew || this.buildingSnapshotIndex > latestSnapshotIndex) break block9;
                    logger.info("[Node {}] Already built a snapshot covering log index {}", (Object)node.getClusterGroup().getCurrentAddress(), (Object)this.buildingSnapshotIndex);
                    snapshotIndex[0] = latestSnapshotIndex;
                    File file = node.getClusterGroup().getStateMachine().getLatestSnapshot();
                    node.getClusterGroup().unlockSnapshotBuild();
                    return file;
                }
                catch (Throwable throwable) {
                    node.getClusterGroup().unlockSnapshotBuild();
                    throw throwable;
                }
            }
            logger.info("[Node {}] Starting build procedure for snapshot covering log index {}", (Object)node.getClusterGroup().getCurrentAddress(), (Object)this.buildingSnapshotIndex);
            long start = System.currentTimeMillis();
            long logIndex = node.getClusterGroup().getTransactionLog().getLastLogIndex();
            logger.info("Building snapshot with last log index {}", (Object)logIndex);
            File snapshot = node.getClusterGroup().getStateMachine().generateSnapshot(logIndex);
            logger.info("Successfully built snapshot with last log index {} in {} seconds", (Object)logIndex, (Object)((System.currentTimeMillis() - start) / 1000L));
            snapshotIndex[0] = logIndex;
            File file = snapshot;
            node.getClusterGroup().unlockSnapshotBuild();
            return file;
        }
        finally {
            node.setBuildingSnapshot(false);
        }
    }

    protected void propagateBackup(RaftNode node, BackupRequest request, StreamObserver<SnapshotData> stream) {
        logger.info("Locked state machine to generate backup");
        StateMachine machine = node.getClusterGroup().getStateMachine();
        machine.generateBackup((List<String>)request.getRepositoriesList(), request.getWithSystemData(), request.getWithRepositoryData(), stream);
        logger.info("Successfully sent backup to node {}", (Object)request.getNodeId());
    }

    protected boolean isStateValid(RaftNode node) {
        NodeState state = node.getState();
        if (state == NodeState.OUT_OF_SYNC || state == NodeState.RESTRICTED) {
            logger.warn("Node {} is in {} state and cannot replicate snapshot to another node", (Object)node.getAddress(), (Object)state);
            return false;
        }
        return true;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void requestS3Backup(S3BackupRequest request, StreamObserver<S3BackupResponse> responseObserver) {
        boolean isBackupSuccessful = false;
        try {
            RaftNode node = this.fetchAndValidateNode(CLOUD_BACKUP);
            this.lockNode(node, CLOUD_BACKUP);
            try {
                logger.info("Node[{}] is creating s3 cloud backup.", (Object)request.getBackupOptions().getNodeId());
                this.validateNode(node, CLOUD_BACKUP);
                isBackupSuccessful = this.propagateS3CloudBackup(node, request);
                responseObserver.onNext((Object)S3BackupResponse.newBuilder().setSuccess(true).build());
                responseObserver.onCompleted();
            }
            finally {
                this.unlockNode(node);
            }
        }
        catch (Exception e) {
            if (!isBackupSuccessful) {
                this.logAndThrowException(CLOUD_BACKUP, e, responseObserver);
            }
            logger.warn("Failed after sending backup: ", (Throwable)e);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void requestAzureBackup(AzureBackupRequest request, StreamObserver<AzureBackupResponse> responseObserver) {
        boolean isBackupSuccessful = false;
        try {
            RaftNode node = this.fetchAndValidateNode(CLOUD_BACKUP);
            this.lockNode(node, CLOUD_BACKUP);
            try {
                logger.info("Node [{}] is building azure cloud backup.", (Object)request.getBackupOptions().getNodeId());
                this.validateNode(node, CLOUD_BACKUP);
                isBackupSuccessful = this.propagateAzureCloudBackup(node, request);
                responseObserver.onNext((Object)AzureBackupResponse.newBuilder().setSuccess(true).build());
                responseObserver.onCompleted();
            }
            finally {
                this.unlockNode(node);
            }
        }
        catch (Exception e) {
            if (!isBackupSuccessful) {
                this.logAndThrowException(CLOUD_BACKUP, e, responseObserver);
            }
            logger.warn("Failed after sending backup: ", (Throwable)e);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void requestGoogleBackup(GoogleBackupRequest request, StreamObserver<GoogleBackupResponse> responseObserver) {
        boolean isBackupSuccessful = false;
        try {
            RaftNode node = this.fetchAndValidateNode(CLOUD_BACKUP);
            this.lockNode(node, CLOUD_BACKUP);
            try {
                logger.info("Node [{}] is building google storage cloud backup.", (Object)request.getBackupOptions().getNodeId());
                this.validateNode(node, CLOUD_BACKUP);
                isBackupSuccessful = this.propagateGoogleCloudBackup(node, request);
                responseObserver.onNext((Object)GoogleBackupResponse.newBuilder().setSuccess(true).build());
                responseObserver.onCompleted();
            }
            finally {
                this.unlockNode(node);
            }
        }
        catch (Exception e) {
            if (!isBackupSuccessful) {
                this.logAndThrowException(CLOUD_BACKUP, e, responseObserver);
            }
            logger.warn("Failed after sending backup: ", (Throwable)e);
        }
    }

    protected boolean propagateS3CloudBackup(RaftNode node, S3BackupRequest s3BackupRequest) {
        boolean bl;
        block8: {
            logger.info("Locked state machine to generate backup");
            StateMachine machine = node.getClusterGroup().getStateMachine();
            S3BackupRequest.S3BucketOptions s3GRPCBucketOptions = s3BackupRequest.getS3BucketOptions();
            S3BucketOptions s3bucketOptions = this.convertGRPCS3BucketOptions(s3GRPCBucketOptions);
            S3AsyncClient s3Async = S3Utils.s3AsyncClientBuilder((S3BucketOptions)s3bucketOptions, (boolean)s3GRPCBucketOptions.getIsAccelerated());
            try {
                S3MultipartUploader s3MultipartUploader = new S3MultipartUploader(s3Async, s3bucketOptions);
                BackupRequest backupOptions = s3BackupRequest.getBackupOptions();
                long cloudPartSize = machine.estimateBackupSize((List<String>)backupOptions.getRepositoriesList(), backupOptions.getWithSystemData(), backupOptions.getWithRepositoryData());
                machine.generateBackup((List<String>)backupOptions.getRepositoriesList(), backupOptions.getWithSystemData(), backupOptions.getWithRepositoryData(), this.createCloudOutputStream((CloudMultipartUploader)s3MultipartUploader, cloudPartSize));
                logger.info("Successfully sent backup to cloud");
                bl = true;
                if (s3Async == null) break block8;
            }
            catch (Throwable throwable) {
                try {
                    if (s3Async != null) {
                        try {
                            s3Async.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                catch (Exception e) {
                    throw new RaftRecoveryException(e);
                }
            }
            s3Async.close();
        }
        return bl;
    }

    protected boolean propagateAzureCloudBackup(RaftNode node, AzureBackupRequest azureBackupRequest) {
        boolean bl;
        block8: {
            logger.info("Locked state machine to generate backup");
            StateMachine machine = node.getClusterGroup().getStateMachine();
            AzureBackupRequest.AzureBlobOptions azureGRPCBucketOptions = azureBackupRequest.getAzureBlobOptions();
            AzureBlobOptions azureBlobOptions = this.convertGRPCAzureBlobOptions(azureGRPCBucketOptions);
            BackupRequest backupOptions = azureBackupRequest.getBackupOptions();
            long partSize = machine.estimateBackupSize((List<String>)backupOptions.getRepositoriesList(), backupOptions.getWithSystemData(), backupOptions.getWithRepositoryData());
            BlobOutputStream blobBackupStream = AzureBlobUtils.getBlobOutputSteam((AzureBlobOptions)azureBlobOptions, (long)partSize);
            try {
                machine.generateBackup((List<String>)backupOptions.getRepositoriesList(), backupOptions.getWithSystemData(), backupOptions.getWithRepositoryData(), (OutputStream)blobBackupStream);
                logger.info("Successfully sent backup to cloud");
                bl = true;
                if (blobBackupStream == null) break block8;
            }
            catch (Throwable throwable) {
                try {
                    if (blobBackupStream != null) {
                        try {
                            blobBackupStream.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                catch (Exception e) {
                    throw new RaftRecoveryException(e);
                }
            }
            blobBackupStream.close();
        }
        return bl;
    }

    protected boolean propagateGoogleCloudBackup(RaftNode node, GoogleBackupRequest googleBackupRequest) {
        logger.info("Locked state machine to generate backup");
        StateMachine machine = node.getClusterGroup().getStateMachine();
        GoogleBackupRequest.GoogleBucketOptions googleGRPCBucketOptions = googleBackupRequest.getGoogleBucketOptions();
        GoogleBucketOptions googlebucketOptions = this.convertGRPCGoogleBucketOptions(googleGRPCBucketOptions);
        try {
            GCMultipartUploader gcMultipartUploader = new GCMultipartUploader(googlebucketOptions);
            BackupRequest backupOptions = googleBackupRequest.getBackupOptions();
            long cloudPartSize = machine.estimateBackupSize((List<String>)backupOptions.getRepositoriesList(), backupOptions.getWithSystemData(), backupOptions.getWithRepositoryData());
            machine.generateBackup((List<String>)backupOptions.getRepositoriesList(), backupOptions.getWithSystemData(), backupOptions.getWithRepositoryData(), this.createCloudOutputStream((CloudMultipartUploader)gcMultipartUploader, cloudPartSize));
            logger.info("Successfully sent backup to cloud");
            return true;
        }
        catch (Exception e) {
            throw new RaftRecoveryException(e);
        }
    }

    protected OutputStream createCloudOutputStream(CloudMultipartUploader MultipartUploader, long partSize) throws IOException {
        return new CloudOutputStream(MultipartUploader, partSize);
    }

    @NotNull
    private S3BucketOptions convertGRPCS3BucketOptions(S3BackupRequest.S3BucketOptions gRPCBucketOptions) {
        return new S3BucketOptions(gRPCBucketOptions.getAccessKey(), gRPCBucketOptions.getAccessKeyId(), gRPCBucketOptions.getRegion(), gRPCBucketOptions.getBucketName(), gRPCBucketOptions.getFileKey(), gRPCBucketOptions.getHost());
    }

    @NotNull
    private AzureBlobOptions convertGRPCAzureBlobOptions(AzureBackupRequest.AzureBlobOptions gRPCBlobOptions) {
        return new AzureBlobOptions(gRPCBlobOptions.getStorageAccount(), gRPCBlobOptions.getAccessKey(), gRPCBlobOptions.getBucketName(), gRPCBlobOptions.getFileKey(), gRPCBlobOptions.getSasSignature(), gRPCBlobOptions.getRawQuery());
    }

    @NotNull
    private GoogleBucketOptions convertGRPCGoogleBucketOptions(GoogleBackupRequest.GoogleBucketOptions gRPCBucketOptions) {
        return new GoogleBucketOptions(gRPCBucketOptions.getBucketName(), gRPCBucketOptions.getFileKey(), gRPCBucketOptions.getAuthenticationFile());
    }

    private void validateNode(RaftNode node, String loggingMethodUser) {
        if (node == null || !node.isInitialized()) {
            throw new StatusRuntimeException(Status.FAILED_PRECONDITION.withCause((Throwable)((Object)new RaftRecoveryException("Raft node not initialized yet"))));
        }
        if (!this.isStateValid(node)) {
            throw new StatusRuntimeException(Status.FAILED_PRECONDITION.withCause((Throwable)((Object)new RaftRecoveryException("Node with address " + node.getAddress() + " cannot send " + loggingMethodUser + " as it is in state " + String.valueOf((Object)node.getState())))));
        }
    }

    private void validateRequest(SnapshotRequest request, RaftNode node) {
        if (!request.getTag().isEmpty() ? !node.getClusterGroup().getTagMap().containsKey(request.getTag()) : !node.getClusterGroup().containsNode(request.getNodeId())) {
            throw new StatusRuntimeException(Status.FAILED_PRECONDITION.withCause((Throwable)((Object)new RaftRecoveryException("Secondary cluster with tag " + request.getTag() + " is not registered with primary cluster"))));
        }
    }

    private void lockNode(RaftNode node, String loggingMethodUser) {
        while (!node.getClusterGroup().tryLockSnapshotBuild(1000L)) {
            this.validateNode(node, loggingMethodUser);
        }
        try {
            node.setBuildingSnapshot(true);
        }
        catch (Exception e) {
            try {
                node.getClusterGroup().unlockSnapshotBuild();
            }
            catch (Exception e1) {
                e.addSuppressed(e);
            }
            throw e;
        }
    }

    private void unlockNode(RaftNode node) {
        try {
            node.setBuildingSnapshot(false);
        }
        finally {
            node.getClusterGroup().unlockSnapshotBuild();
        }
    }

    private SnapshotResponse generateResponse(SnapshotResponse.Status status, long lastLogIndex) {
        return SnapshotResponse.newBuilder().setStatus(status).setLastLogIndex(lastLogIndex).build();
    }

    private void logAndThrowException(String failedOperation, Exception e, StreamObserver<?> delegateObserver) {
        StatusRuntimeException exception;
        logger.error("Failed to send {}: ", (Object)failedOperation, (Object)e);
        if (e instanceof StatusRuntimeException) {
            exception = (StatusRuntimeException)e;
        } else {
            Status codeStatus = Status.INTERNAL;
            if (e.getCause() instanceof InsufficientDiskSpaceException) {
                codeStatus = Status.FAILED_PRECONDITION;
            }
            exception = new StatusRuntimeException(codeStatus.withDescription("Failed to send " + failedOperation + " due to: " + e.getMessage()).withCause(e.getCause()));
        }
        if (delegateObserver != null) {
            delegateObserver.onError((Throwable)exception);
        }
    }

    private Map<String, String> buildGroupMapping(ClusterGroup group) {
        HashMap<String, String> map = new HashMap<String, String>();
        for (NodeInfo node : group.getNodes()) {
            map.put(node.getRpcAddress(), node.getHttpAddress());
        }
        return map;
    }

    private Map<String, Integer> buildRepositoryMapping(ClusterGroup group) {
        return group.getTransactionLog().getChannelMapping();
    }

    private static class RecoveryHistory
    extends LinkedHashMap<Long, AtomicInteger> {
        private static final int CAPACITY = 10;

        public RecoveryHistory() {
            super(10, 0.95f, true);
        }

        public boolean recordAccess(Long snapshot) {
            boolean forceReplication;
            AtomicInteger counter = this.computeIfAbsent(snapshot, k -> new AtomicInteger());
            boolean bl = forceReplication = counter.incrementAndGet() > RECOVERY_ATTEMPTS;
            if (forceReplication) {
                counter.set(0);
            }
            return forceReplication;
        }

        @Override
        protected boolean removeEldestEntry(Map.Entry<Long, AtomicInteger> eldest) {
            return this.size() > 10;
        }
    }
}

