/*
 * Decompiled with CFR 0.152.
 */
package com.ontotext.forest.recovery;

import com.azure.core.exception.AzureException;
import com.azure.identity.CredentialUnavailableException;
import com.azure.storage.blob.BlobClient;
import com.azure.storage.blob.models.BlobStorageException;
import com.azure.storage.blob.specialized.BlobOutputStream;
import com.google.common.annotations.VisibleForTesting;
import com.ontotext.forest.core.semantic.SemanticDataManagement;
import com.ontotext.forest.recovery.BackupOperationStatus;
import com.ontotext.graphdb.GraphDBHTTPContext;
import com.ontotext.graphdb.GraphDBRepositoryManager;
import com.ontotext.graphdb.cloud.CloudMultipartUploader;
import com.ontotext.graphdb.cloud.CloudOutputStream;
import com.ontotext.graphdb.cloud.aws.S3BucketOptions;
import com.ontotext.graphdb.cloud.aws.S3MultipartDownloader;
import com.ontotext.graphdb.cloud.aws.S3MultipartUploader;
import com.ontotext.graphdb.cloud.aws.S3Utils;
import com.ontotext.graphdb.cloud.azure.AzureBlobOptions;
import com.ontotext.graphdb.cloud.azure.AzureBlobUtils;
import com.ontotext.graphdb.cloud.azure.AzureMultipartDownloader;
import com.ontotext.graphdb.cloud.google.GCMultipartDownloader;
import com.ontotext.graphdb.cloud.google.GCMultipartUploader;
import com.ontotext.graphdb.cloud.google.GoogleBucketOptions;
import com.ontotext.graphdb.raft.grpc.AzureBackupRequest;
import com.ontotext.graphdb.raft.grpc.BackupRequest;
import com.ontotext.graphdb.raft.grpc.GoogleBackupRequest;
import com.ontotext.graphdb.raft.grpc.NodeException;
import com.ontotext.graphdb.raft.grpc.ReportingIterator;
import com.ontotext.graphdb.raft.grpc.RpcNodeClient;
import com.ontotext.graphdb.raft.grpc.S3BackupRequest;
import com.ontotext.graphdb.raft.grpc.SnapshotData;
import com.ontotext.graphdb.recovery.BackupException;
import com.ontotext.graphdb.recovery.RecoveryException;
import com.ontotext.graphdb.recovery.RecoveryUtil;
import com.ontotext.graphdb.recovery.SnapshotMetadata;
import com.ontotext.graphdb.recovery.SnapshotOptions;
import com.ontotext.graphdb.recovery.SnapshotValidationException;
import com.ontotext.raft.GraphDBReplicationCluster;
import com.ontotext.raft.repository.ClusterRepositoryManager;
import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URI;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.BasicFileAttributes;
import java.security.GeneralSecurityException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.atomic.AtomicReference;
import java.util.stream.Collectors;
import org.apache.commons.fileupload2.core.FileItemInput;
import org.apache.commons.lang3.StringUtils;
import org.eclipse.rdf4j.query.UpdateExecutionException;
import org.eclipse.rdf4j.repository.manager.RepositoryManager;
import org.jetbrains.annotations.NotNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import software.amazon.awssdk.services.s3.S3AsyncClient;
import software.amazon.awssdk.services.s3.S3Client;

@Service
public class RecoveryService {
    private static final Logger logger = LoggerFactory.getLogger(RecoveryService.class);
    public static final String SNAPSHOT_FILE_NAME = "recovery-snapshot-";
    protected static final String CANNOT_CREATE_BACKUP_ANOTHER_BACKUP_PROCESS_IS_ALREADY_RUNNING = "Cannot create backup. Another recovery process is already running.";
    protected static final String CANNOT_RESTORE_FROM_BACKUP_ANOTHER_BACKUP_PROCESS_IS_ALREADY_RUNNING = "Cannot apply backup. Another recovery process is already running.";
    private final SemanticDataManagement semanticDataManagement;
    private final Path snapshotDir = RecoveryUtil.resolveSnapshotDir();
    private static final long CLIENT_TIMEOUT = 10000L;
    public static final String EXTENSION = ".tar";
    private final AtomicReference<RecoveryStatus> recoveryOperationInProgress = new AtomicReference();
    private final AtomicReference<BackupOperationStatus> backupOperationStatus = new AtomicReference();
    private SnapshotMetadata snapshotMetadata;

    @Autowired
    public RecoveryService(SemanticDataManagement semanticDataManagement) {
        this.semanticDataManagement = semanticDataManagement;
    }

    public File createSnapshot(String id, SnapshotOptions options) throws BackupException, IOException {
        GraphDBRepositoryManager graphDbManager = this.getGraphDBRepositoryManager();
        RecoveryUtil.validateRepositoriesPartialCreate((GraphDBRepositoryManager)graphDbManager, (SnapshotOptions)options);
        File source = graphDbManager.getBaseDir();
        File dest = this.getSnapshotDestination(id);
        RecoveryUtil.createSnapshot((File)source, (File)dest, (SnapshotOptions)options, (GraphDBRepositoryManager)graphDbManager);
        return dest;
    }

    public void createBackup(SnapshotOptions options, OutputStream stream) throws BackupException, IOException {
        if (this.recoveryOperationInProgress.compareAndSet(null, RecoveryStatus.CREATE_BACKUP_IN_PROGRESS)) {
            try {
                this.setRecoveryOperationInfo(BackupOperationStatus.BackupOperationInProgress.CREATE_BACKUP_IN_PROGRESS, options);
                this.createBackupUnprotected(options, stream);
            }
            finally {
                this.resetRecoveryOperation();
            }
        } else {
            throw new BackupException(CANNOT_CREATE_BACKUP_ANOTHER_BACKUP_PROCESS_IS_ALREADY_RUNNING);
        }
    }

    private void resetRecoveryOperation() {
        this.clearRecoveryOperationInfo();
        this.recoveryOperationInProgress.set(null);
    }

    void createBackupUnprotected(SnapshotOptions options, OutputStream stream) throws BackupException, IOException {
        logger.info("Received backup request");
        long start = System.currentTimeMillis();
        if (this.inCluster()) {
            GraphDBReplicationCluster cluster = ((ClusterRepositoryManager)this.getGraphDBRepositoryManager()).getReplicationCluster();
            if (!cluster.isWritable()) {
                throw new BackupException("Cluster is not synchronized at the moment");
            }
            RecoveryUtil.validateRepositoriesPartialCreate((GraphDBRepositoryManager)this.getGraphDBRepositoryManager(), (SnapshotOptions)options);
            RpcNodeClient client = this.getNodeClient(cluster);
            String nodeAddress = client.getAddress();
            logger.info("Sending backup request to node {}", (Object)nodeAddress);
            this.getBackupOperationStatus().setAdditionalClusterNodeBackUpInfo(nodeAddress);
            ReportingIterator backup = client.requestBackup(options.isWithSystemData(), options.isWithRepositoryData(), options.getRepositories());
            try {
                while (backup.hasNext()) {
                    SnapshotData next = (SnapshotData)backup.next();
                    if (!next.getSuccess()) {
                        throw new BackupException("Node with address " + client.getAddress() + " was unable to create backup");
                    }
                    if (next.getData().isEmpty()) continue;
                    stream.write(next.getData().toByteArray());
                }
            }
            catch (Exception e) {
                if (!(e instanceof NodeException)) {
                    backup.reportError("Error occurred on leader node", (Throwable)e);
                }
                throw e;
            }
        } else {
            GraphDBRepositoryManager graphDbManager = this.getGraphDBRepositoryManager();
            RecoveryUtil.validateRepositoriesPartialCreate((GraphDBRepositoryManager)graphDbManager, (SnapshotOptions)options);
            File source = graphDbManager.getBaseDir();
            RecoveryUtil.createSnapshot((File)source, (SnapshotOptions)options, (GraphDBRepositoryManager)graphDbManager, (OutputStream)stream);
        }
        logger.info("Uploaded backup successfully in {} seconds", (Object)((System.currentTimeMillis() - start) / 1000L));
    }

    public RpcNodeClient getNodeClient(GraphDBReplicationCluster cluster) throws BackupException {
        long lastIndex = cluster.getClusterGroup().getTransactionLog().getLastLogIndex();
        long start = System.currentTimeMillis();
        while (System.currentTimeMillis() - start < 10000L) {
            for (RpcNodeClient client : cluster.getClusterGroup()) {
                if (client.getMatchIndex() != lastIndex || client.getStatus() != RpcNodeClient.Status.IN_SYNC) continue;
                return client;
            }
        }
        throw new BackupException("Unable to create backup in cluster setup for log index " + lastIndex + " due to timeout");
    }

    public void applyBackup(InputStream backupStream, SnapshotOptions snapshotOptions) throws BackupException {
        if (this.recoveryOperationInProgress.compareAndSet(null, RecoveryStatus.APPLY_BACKUP_IN_PROGRESS)) {
            try {
                this.setRecoveryOperationInfo(BackupOperationStatus.BackupOperationInProgress.RESTORE_BACKUP_IN_PROGRESS, snapshotOptions);
                this.applyBackupUnprotected(backupStream, snapshotOptions);
            }
            finally {
                this.resetRecoveryOperation();
            }
        } else {
            throw new BackupException(CANNOT_RESTORE_FROM_BACKUP_ANOTHER_BACKUP_PROCESS_IS_ALREADY_RUNNING);
        }
    }

    private void applyBackupUnprotected(InputStream backupStream, SnapshotOptions snapshotOptions) throws BackupException {
        if (this.inCluster()) {
            this.applyBackupInCluster(backupStream, snapshotOptions);
        } else {
            logger.info("Recovering from backup");
            File restoreSnapshotFile = null;
            try {
                restoreSnapshotFile = this.getSnapshotDestination(SNAPSHOT_FILE_NAME + String.valueOf(UUID.randomUUID()));
                this.validateSnapshotAndWriteTo(backupStream, snapshotOptions, restoreSnapshotFile);
                this.applySnapshot(restoreSnapshotFile.getName(), snapshotOptions);
            }
            catch (Exception e) {
                logger.error("Could not recover from backup due to: ", (Throwable)e);
                if (e instanceof BackupException) {
                    throw (BackupException)e;
                }
                throw new BackupException(e.getMessage());
            }
            finally {
                if (restoreSnapshotFile != null && restoreSnapshotFile.exists() && !restoreSnapshotFile.delete()) {
                    logger.warn("Unable to automatically remove snapshot file {} after recovery procedure", (Object)restoreSnapshotFile.getAbsolutePath());
                }
            }
        }
    }

    protected boolean inCluster() {
        return this.getGraphDBRepositoryManager() instanceof ClusterRepositoryManager;
    }

    private void applyBackupInCluster(InputStream backupStream, SnapshotOptions snapshotOptions) throws BackupException {
        logger.info("Recovering from backup in cluster");
        GraphDBReplicationCluster cluster = ((ClusterRepositoryManager)this.getGraphDBRepositoryManager()).getReplicationCluster();
        if (!cluster.isWritable() || !cluster.isPrimaryCluster()) {
            throw new BackupException("Cluster is not writable at the moment");
        }
        try {
            this.validateClusterSnapshotAndWriteTo(backupStream, cluster, snapshotOptions, this.getGraphDBRepositoryManager());
        }
        catch (Exception e) {
            logger.error("Rolling back record of backup due to: ", (Throwable)e);
            if (!(e instanceof SnapshotValidationException)) {
                cluster.rollbackBackup();
            }
            throw new BackupException((Throwable)e);
        }
        try {
            if (cluster.applyAndReplicateBackup("" + snapshotOptions.hashCode()) < 1L) {
                throw new BackupException("Could not restore from backup as cluster is not synced");
            }
        }
        catch (Exception e) {
            if (e instanceof UpdateExecutionException) {
                cluster.rollbackBackup();
            }
            logger.error("Could not apply backup due to: ", (Throwable)e);
            throw new BackupException((Throwable)e);
        }
    }

    @NotNull
    protected File getSnapshotDestination(String id) throws BackupException {
        File dest = new File(this.snapshotDir.toFile(), this.getSnapshotFileName(id));
        if (dest.exists() || dest.isDirectory()) {
            throw new BackupException(String.format("Backup %s already exists.", id));
        }
        dest.getParentFile().mkdirs();
        return dest;
    }

    @NotNull
    public GraphDBRepositoryManager getGraphDBRepositoryManager() {
        RepositoryManager manager = this.semanticDataManagement.getCurrentLocationOrThrow().sesameManager();
        if (!(manager instanceof GraphDBRepositoryManager)) {
            throw new IllegalStateException("Cannot create backup of GraphDB with different type");
        }
        return (GraphDBRepositoryManager)manager;
    }

    @Deprecated(forRemoval=true)
    public void updateSnapshot(String name, SnapshotOptions options) throws BackupException, IOException {
        File tempSnapshot = this.createSnapshot(name + "-temp", options);
        File dest = this.createSnapshotFile(name);
        try {
            this.deleteSnapshotOrThrow(name);
        }
        catch (BackupException backupException) {
            // empty catch block
        }
        Files.move(tempSnapshot.toPath(), dest.toPath(), StandardCopyOption.ATOMIC_MOVE);
    }

    public void applySnapshot(String id, SnapshotOptions options) throws BackupException, IOException {
        File snapshotFile = this.getSnapshotFileOrThrowIfNotExists(id);
        GraphDBRepositoryManager graphDbManager = this.getGraphDBRepositoryManager();
        try {
            RecoveryUtil.applySnapshot((File)snapshotFile, (SnapshotOptions)options, (GraphDBRepositoryManager)graphDbManager);
        }
        catch (RecoveryException e) {
            throw new BackupException(e.getMessage());
        }
    }

    @NotNull
    public File getSnapshotFileOrThrowIfNotExists(String id) throws BackupException {
        File source = this.createSnapshotFile(id);
        if (!source.exists() || source.isDirectory()) {
            throw new BackupException("Backup not found: " + id, true);
        }
        return source;
    }

    public List<Map<String, Object>> getSnapshots() {
        if (!this.snapshotDir.toFile().exists()) {
            return new LinkedList<Map<String, Object>>();
        }
        File[] files = this.snapshotDir.toFile().listFiles();
        if (files == null) {
            return new LinkedList<Map<String, Object>>();
        }
        return Arrays.stream(files).map(file -> {
            HashMap<String, Object> fileMap = new HashMap<String, Object>();
            String name = file.getName().toLowerCase(Locale.ROOT).endsWith(EXTENSION) ? file.getName().substring(0, file.getName().length() - 4) : file.getName();
            fileMap.put("name", name);
            try {
                SnapshotMetadata snapshotMetaData = RecoveryUtil.getSnapshotMetadata((File)this.getSnapshotFileOrThrowIfNotExists(name));
                fileMap.put("size", snapshotMetaData.getSize());
                if (this.checkSnapshotIsPartial(snapshotMetaData)) {
                    fileMap.put("repositories", snapshotMetaData.getRepositories());
                    fileMap.put("withSystemData", snapshotMetaData.isWithUserData());
                }
            }
            catch (Exception e) {
                logger.error("Could not get backup metadata for: {}", file);
            }
            try {
                BasicFileAttributes attr = Files.readAttributes(file.toPath(), BasicFileAttributes.class, new LinkOption[0]);
                fileMap.put("dateCreated", attr.creationTime().toString());
                if (!fileMap.containsKey("size")) {
                    fileMap.put("size", attr.size());
                }
            }
            catch (IOException e) {
                logger.error("Could not get attributes for: {}", file);
            }
            return fileMap;
        }).collect(Collectors.toList());
    }

    @Deprecated(forRemoval=true)
    private boolean checkSnapshotIsPartial(SnapshotMetadata metadata) {
        return metadata.getRepositories() != null && !metadata.getRepositories().isEmpty();
    }

    public void deleteSnapshotOrThrow(String name) throws IOException, BackupException {
        File source = this.getSnapshotFileOrThrowIfNotExists(name);
        Files.delete(source.toPath());
    }

    public File createSnapshotFile(String name) {
        return new File(this.snapshotDir.toFile(), this.getSnapshotFileName(name));
    }

    @NotNull
    protected String getSnapshotFileName(String id) {
        return StringUtils.appendIfMissing((String)id, (CharSequence)EXTENSION, (CharSequence[])new CharSequence[0]);
    }

    protected void validateSnapshotAndWriteTo(InputStream backupStream, SnapshotOptions snapshotOptions, File snapshotFile) throws BackupException {
        try {
            BufferedOutputStream output = new BufferedOutputStream(new FileOutputStream(snapshotFile));
            RecoveryUtil.validateAndRecordBackup((InputStream)new BufferedInputStream(backupStream), (SnapshotOptions)snapshotOptions, x -> output, (GraphDBRepositoryManager)this.getGraphDBRepositoryManager(), (boolean)false, this::setSnapshotMetadata);
        }
        catch (RecoveryException | IOException e) {
            logger.error("Invalid snapshot file", e);
            throw new BackupException(e.getMessage());
        }
    }

    protected void validateClusterSnapshotAndWriteTo(InputStream backupStream, GraphDBReplicationCluster cluster, SnapshotOptions options, GraphDBRepositoryManager repositoryManager) throws BackupException, RecoveryException {
        try {
            RecoveryUtil.validateAndRecordBackup((InputStream)new BufferedInputStream(backupStream), (SnapshotOptions)options, affectedRepos -> cluster.recordBackup(new ArrayList(affectedRepos), options.isCleanDataDir()), (GraphDBRepositoryManager)repositoryManager, (boolean)true, this::setSnapshotMetadata);
        }
        catch (RecoveryException e) {
            logger.error("Invalid snapshot file", (Throwable)e);
            if (e instanceof SnapshotValidationException) {
                throw e;
            }
            throw new BackupException(e.getMessage());
        }
    }

    public File getSnapshotDir() {
        File file = this.snapshotDir.toFile();
        if (!file.exists()) {
            file.mkdirs();
        }
        return file;
    }

    public void applyCloudBackup(URI bucketUri, SnapshotOptions snapshotOptions, FileItemInput authenticationFile) throws BackupException {
        if (this.recoveryOperationInProgress.compareAndSet(null, RecoveryStatus.APPLY_BACKUP_IN_PROGRESS)) {
            try {
                String bucketType;
                this.setRecoveryOperationInfo(BackupOperationStatus.BackupOperationInProgress.RESTORE_CLOUD_BACKUP_IN_PROGRESS, snapshotOptions);
                switch (bucketType = Objects.isNull(bucketUri.getScheme()) ? "" : bucketUri.getScheme()) {
                    case "s3": {
                        S3BucketOptions bucketOptions = new S3BucketOptions(bucketUri);
                        this.applyS3Backup(bucketOptions, snapshotOptions);
                        return;
                    }
                    case "az": {
                        AzureBlobOptions bucketOptions = new AzureBlobOptions(bucketUri);
                        this.applyAzureBackup(bucketOptions, snapshotOptions);
                        return;
                    }
                    case "gs": {
                        GoogleBucketOptions bucketOptions = new GoogleBucketOptions(bucketUri, authenticationFile);
                        this.applyGoogleBackup(bucketOptions, snapshotOptions);
                        return;
                    }
                }
                try {
                    BackupException ex = new BackupException(String.format("Unsupported bucket type: %s, currently only \"s3\", \"gs\" buckets and \"az\" containers are supported", bucketUri.getScheme()));
                    logger.error("Restore failed due to an error: ", (Throwable)ex);
                    throw ex;
                }
                catch (Exception e) {
                    if (e instanceof BackupException) {
                        throw (BackupException)((Object)e);
                    }
                    logger.error("Could not retrieve cloud backup due to: ", (Throwable)e);
                    throw new BackupException(e.getMessage());
                }
            }
            finally {
                this.resetRecoveryOperation();
            }
        }
        throw new BackupException(CANNOT_RESTORE_FROM_BACKUP_ANOTHER_BACKUP_PROCESS_IS_ALREADY_RUNNING);
    }

    public void applyCloudBackup(URI bucketUri, SnapshotOptions snapshotOptions) throws BackupException {
        this.applyCloudBackup(bucketUri, snapshotOptions, null);
    }

    private void applyS3Backup(S3BucketOptions bucketOptions, SnapshotOptions snapshotOptions) throws BackupException, IOException {
        try (S3AsyncClient s3 = S3Utils.s3AsyncClientBuilder((S3BucketOptions)bucketOptions, (boolean)this.isS3AccelerationEnabled(bucketOptions));
             S3MultipartDownloader amazonStream = new S3MultipartDownloader(s3, bucketOptions);){
            logger.info("Fetching backup file: {}", (Object)bucketOptions.getFileKey());
            logger.info("From bucket: {}", (Object)bucketOptions.getBucketName());
            logger.info("On bucket host: {}", bucketOptions.getHostAsUri() != null ? String.valueOf(bucketOptions.getHostAsUri()) + "/" : "Amazon Web Services");
            this.applyBackupUnprotected(amazonStream.getPipedInputStream(), snapshotOptions);
        }
        catch (InterruptedException | GeneralSecurityException | ExecutionException e) {
            logger.error("Restore failed due to: ", (Throwable)e);
            throw new BackupException(e.getMessage());
        }
    }

    private void applyAzureBackup(AzureBlobOptions bucketOptions, SnapshotOptions snapshotOptions) throws BackupException, IOException {
        BlobClient azureClient = AzureBlobUtils.getAzureContainerClient((AzureBlobOptions)bucketOptions).getBlobClient(bucketOptions.getFileKey());
        try (AzureMultipartDownloader blobStream = new AzureMultipartDownloader(azureClient);){
            logger.info("Fetching backup file: {}", (Object)bucketOptions.getFileKey());
            logger.info("From bucket: {}", (Object)bucketOptions.getContainerName());
            this.applyBackupUnprotected(blobStream.getPipedInputStream(), snapshotOptions);
        }
        catch (CredentialUnavailableException | BlobStorageException ex) {
            AzureBlobUtils.transformAzureException((AzureException)ex);
        }
        catch (GeneralSecurityException e) {
            logger.error("Restore failed due to: ", (Throwable)e);
            throw new BackupException(e.getMessage());
        }
    }

    private void applyGoogleBackup(GoogleBucketOptions bucketOptions, SnapshotOptions snapshotOptions) throws IOException, GeneralSecurityException, BackupException {
        try (GCMultipartDownloader chunkStream = new GCMultipartDownloader(bucketOptions);){
            logger.info("Fetching backup file: {}", (Object)bucketOptions.getFileKey());
            logger.info("From bucket: {}", (Object)bucketOptions.getBucketName());
            this.applyBackupUnprotected(chunkStream.getPipedInputStream(), snapshotOptions);
        }
    }

    public void createCloudBackup(URI bucketUri, SnapshotOptions backupOptions, FileItemInput authenticationFile) throws BackupException {
        if (this.recoveryOperationInProgress.compareAndSet(null, RecoveryStatus.CREATE_BACKUP_IN_PROGRESS)) {
            try {
                String bucketType;
                this.setRecoveryOperationInfo(BackupOperationStatus.BackupOperationInProgress.CREATE_CLOUD_BACKUP_IN_PROGRESS, backupOptions);
                switch (bucketType = Objects.isNull(bucketUri.getScheme()) ? "" : bucketUri.getScheme()) {
                    case "s3": {
                        S3BucketOptions bucketOptions = new S3BucketOptions(bucketUri);
                        this.createS3Backup(bucketOptions, backupOptions);
                        return;
                    }
                    case "az": {
                        AzureBlobOptions blobOptions = new AzureBlobOptions(bucketUri);
                        this.createAzureBackup(blobOptions, backupOptions);
                        return;
                    }
                    case "gs": {
                        GoogleBucketOptions bucketOptions = new GoogleBucketOptions(bucketUri, authenticationFile);
                        this.createGoogleBackup(bucketOptions, backupOptions);
                        return;
                    }
                }
                try {
                    BackupException ex = new BackupException(String.format("Unsupported bucket type: %s, currently only \"s3\", \"gs\" buckets and \"az\" containers are supported", bucketUri.getScheme()));
                    logger.error("Creating cloud backup failed due to: ", (Throwable)ex);
                    throw ex;
                }
                catch (Throwable e) {
                    if (e instanceof BackupException) {
                        throw (BackupException)e;
                    }
                    logger.error("Creating cloud backup failed due to: ", e);
                    throw new BackupException(e.getMessage());
                }
            }
            finally {
                this.resetRecoveryOperation();
            }
        }
        throw new BackupException(CANNOT_CREATE_BACKUP_ANOTHER_BACKUP_PROCESS_IS_ALREADY_RUNNING);
    }

    public void createCloudBackup(URI bucketUri, SnapshotOptions backupOptions) throws BackupException {
        this.createCloudBackup(bucketUri, backupOptions, null);
    }

    private void createS3Backup(S3BucketOptions bucketOptions, SnapshotOptions snapshotOptions) throws BackupException {
        this.validates3BucketOptions(bucketOptions);
        boolean isAccelerationEnabled = this.isS3AccelerationEnabled(bucketOptions);
        if (this.inCluster()) {
            this.s3BackupAndUploadInCluster(bucketOptions, snapshotOptions, isAccelerationEnabled);
            return;
        }
        long partSize = RecoveryUtil.checkS3ReposAndEstimatePartSize((SnapshotOptions)snapshotOptions, (GraphDBRepositoryManager)this.getGraphDBRepositoryManager());
        try (S3AsyncClient s3Async = S3Utils.s3AsyncClientBuilder((S3BucketOptions)bucketOptions, (boolean)isAccelerationEnabled);){
            S3MultipartUploader multipartUploadHelper = new S3MultipartUploader(s3Async, bucketOptions);
            try (CloudOutputStream backupStream = new CloudOutputStream((CloudMultipartUploader)multipartUploadHelper, partSize);){
                this.S3BackupAndUpload(snapshotOptions, bucketOptions, backupStream);
            }
            catch (IOException e) {
                logger.error("Could not create backup due to: ", (Throwable)e);
                throw new BackupException(e.getMessage());
            }
        }
    }

    private void S3BackupAndUpload(SnapshotOptions snapshotOptions, S3BucketOptions bucketOptions, CloudOutputStream out) throws BackupException {
        try {
            this.loggingMessagesForStartingS3BackupAndUpload(bucketOptions);
            this.createBackupUnprotected(snapshotOptions, (OutputStream)out);
        }
        catch (IOException e) {
            logger.error("Could not create backup due to: ", (Throwable)e);
            throw new BackupException(e.getMessage());
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void s3BackupAndUploadInCluster(S3BucketOptions bucketOptions, SnapshotOptions snapshotOptions, boolean isAccelerationEnabled) throws BackupException {
        try {
            this.loggingMessagesForStartingS3BackupAndUpload(bucketOptions);
            GraphDBReplicationCluster cluster = ((ClusterRepositoryManager)this.getGraphDBRepositoryManager()).getReplicationCluster();
            if (!cluster.isWritable()) {
                throw new BackupException("Cluster is not synchronized at the moment");
            }
            RpcNodeClient client = this.getNodeClient(cluster);
            BackupRequest.Builder backupRequestBuilder = BackupRequest.newBuilder().setWithSystemData(snapshotOptions.isWithSystemData()).setWithRepositoryData(snapshotOptions.isWithRepositoryData()).addAllRepositories((Iterable)(snapshotOptions.getRepositories() == null ? List.of() : snapshotOptions.getRepositories())).setNodeId(client.getAddress());
            S3BackupRequest.S3BucketOptions.Builder s3BucketOptions = S3BackupRequest.S3BucketOptions.newBuilder().setBucketName(bucketOptions.getBucketName()).setFileKey(bucketOptions.getFileKey()).setIsAccelerated(isAccelerationEnabled);
            if (!StringUtils.isBlank((CharSequence)bucketOptions.getRegion())) {
                s3BucketOptions.setRegion(bucketOptions.getRegion());
            }
            if (!StringUtils.isBlank((CharSequence)bucketOptions.getAccessKey()) && !StringUtils.isBlank((CharSequence)bucketOptions.getAccessKeyId())) {
                s3BucketOptions.setAccessKey(bucketOptions.getAccessKey()).setAccessKeyId(bucketOptions.getAccessKeyId());
            }
            if (!StringUtils.isBlank((CharSequence)bucketOptions.getHost())) {
                s3BucketOptions.setHost(bucketOptions.getHost());
            }
            S3BackupRequest s3BackupRequest = S3BackupRequest.newBuilder().setBackupOptions(backupRequestBuilder).setS3BucketOptions(s3BucketOptions).build();
            String nodeAddress = client.getAddress();
            logger.info("Sending backup request to node {}", (Object)nodeAddress);
            this.getBackupOperationStatus().setAdditionalClusterNodeBackUpInfo(nodeAddress);
            if (client.requestCloudBackup(s3BackupRequest).getSuccess()) {
                logger.info("Cloud backup successfully performed.");
            }
        }
        finally {
            this.recoveryOperationInProgress.set(null);
        }
    }

    private void validates3BucketOptions(S3BucketOptions s3bucketOptions) throws BackupException {
        HashMap<String, String> s3variables = new HashMap<String, String>();
        if (StringUtils.isBlank((CharSequence)s3bucketOptions.getAccessKey()) || StringUtils.isBlank((CharSequence)s3bucketOptions.getAccessKeyId())) {
            logger.debug("Missing explicitly set credentials, will default to AWS SDK credential provider");
        }
        s3variables.put("bucket name", s3bucketOptions.getBucketName());
        StringBuilder sb = new StringBuilder();
        for (Map.Entry s3param : s3variables.entrySet()) {
            if (!StringUtils.isBlank((CharSequence)((CharSequence)s3param.getValue()))) continue;
            if (sb.length() > 0) {
                sb.append(", ");
            }
            sb.append((String)s3param.getKey());
        }
        if (sb.length() > 0) {
            throw new BackupException("S3 cloud backup is missing parameter: " + String.valueOf(sb));
        }
    }

    private void createAzureBackup(AzureBlobOptions blobOptions, SnapshotOptions snapshotOptions) throws BackupException {
        if (this.inCluster()) {
            this.AzureBackupAndUploadInCluster(blobOptions, snapshotOptions);
            return;
        }
        long partSize = RecoveryUtil.checkAzureReposAndEstimatePartSize((SnapshotOptions)snapshotOptions, (GraphDBRepositoryManager)this.getGraphDBRepositoryManager());
        try (BlobOutputStream backupStream = AzureBlobUtils.getBlobOutputSteam((AzureBlobOptions)blobOptions, (long)partSize);){
            this.AzureBackupAndUpload(snapshotOptions, blobOptions, backupStream);
        }
        catch (IOException e) {
            logger.error("Could not create backup due to: ", (Throwable)e);
            throw new BackupException(e.getMessage());
        }
    }

    private void AzureBackupAndUpload(SnapshotOptions snapshotOptions, AzureBlobOptions bucketOptions, BlobOutputStream out) throws BackupException {
        try {
            this.loggingMessagesForStartingAzureBackupAndUpload(bucketOptions);
            this.createBackupUnprotected(snapshotOptions, (OutputStream)out);
        }
        catch (IOException e) {
            logger.error("Could not create backup due to: ", (Throwable)e);
            throw new BackupException(e.getMessage());
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void AzureBackupAndUploadInCluster(AzureBlobOptions bucketOptions, SnapshotOptions snapshotOptions) throws BackupException {
        try {
            this.loggingMessagesForStartingAzureBackupAndUpload(bucketOptions);
            GraphDBReplicationCluster cluster = ((ClusterRepositoryManager)this.getGraphDBRepositoryManager()).getReplicationCluster();
            if (!cluster.isWritable()) {
                throw new BackupException("Cluster is not synchronized at the moment");
            }
            RpcNodeClient client = this.getNodeClient(cluster);
            BackupRequest.Builder backupRequestBuilder = BackupRequest.newBuilder().setWithSystemData(snapshotOptions.isWithSystemData()).setWithRepositoryData(snapshotOptions.isWithRepositoryData()).addAllRepositories((Iterable)(snapshotOptions.getRepositories() == null ? List.of() : snapshotOptions.getRepositories())).setNodeId(client.getAddress());
            AzureBackupRequest.AzureBlobOptions.Builder azureBlobOptions = AzureBackupRequest.AzureBlobOptions.newBuilder().setStorageAccount(bucketOptions.getStorageAccount()).setBucketName(bucketOptions.getContainerName()).setFileKey(bucketOptions.getFileKey());
            if (!StringUtils.isBlank((CharSequence)bucketOptions.getAccessKey())) {
                azureBlobOptions.setAccessKey(bucketOptions.getAccessKey());
            }
            if (!StringUtils.isBlank((CharSequence)bucketOptions.getSasSignature())) {
                azureBlobOptions.setSasSignature(bucketOptions.getSasSignature());
                azureBlobOptions.setRawQuery(bucketOptions.getRawQuery());
            }
            AzureBackupRequest azureBackupRequest = AzureBackupRequest.newBuilder().setBackupOptions(backupRequestBuilder).setAzureBlobOptions(azureBlobOptions).build();
            String nodeAddress = client.getAddress();
            logger.info("Sending backup request to node {}", (Object)nodeAddress);
            this.backupOperationStatus.get().setAdditionalClusterNodeBackUpInfo(nodeAddress);
            if (client.requestAzureBackup(azureBackupRequest).getSuccess()) {
                logger.info("Cloud backup successfully performed.");
            }
        }
        finally {
            this.recoveryOperationInProgress.set(null);
        }
    }

    private void createGoogleBackup(GoogleBucketOptions bucketOptions, SnapshotOptions snapshotOptions) throws BackupException {
        this.loggingMessagesForStartingGoogleBackupAndUpload(bucketOptions);
        if (this.inCluster()) {
            this.GCBackupAndUploadInCluster(bucketOptions, snapshotOptions);
            return;
        }
        long partSize = RecoveryUtil.checkGCloudReposAndEstimatePartSize((SnapshotOptions)snapshotOptions, (GraphDBRepositoryManager)this.getGraphDBRepositoryManager());
        GCMultipartUploader multipartUploader = new GCMultipartUploader(bucketOptions);
        try (CloudOutputStream backupStream = new CloudOutputStream((CloudMultipartUploader)multipartUploader, partSize);){
            this.GCBackupAndUpload(snapshotOptions, backupStream);
        }
        catch (IOException e) {
            logger.error("Could not create backup due to: ", (Throwable)e);
            throw new BackupException(e.getMessage());
        }
    }

    private void GCBackupAndUpload(SnapshotOptions snapshotOptions, CloudOutputStream out) throws BackupException {
        try {
            this.createBackupUnprotected(snapshotOptions, (OutputStream)out);
        }
        catch (IOException e) {
            logger.error("Could not create backup due to: ", (Throwable)e);
            throw new BackupException(e.getMessage());
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void GCBackupAndUploadInCluster(GoogleBucketOptions bucketOptions, SnapshotOptions snapshotOptions) throws BackupException {
        try {
            GraphDBReplicationCluster cluster = ((ClusterRepositoryManager)this.getGraphDBRepositoryManager()).getReplicationCluster();
            if (!cluster.isWritable()) {
                throw new BackupException("Cluster is not synchronized at the moment");
            }
            RpcNodeClient client = this.getNodeClient(cluster);
            BackupRequest.Builder backupRequestBuilder = BackupRequest.newBuilder().setWithSystemData(snapshotOptions.isWithSystemData()).setWithRepositoryData(snapshotOptions.isWithRepositoryData()).addAllRepositories((Iterable)(snapshotOptions.getRepositories() == null ? List.of() : snapshotOptions.getRepositories())).setNodeId(client.getAddress());
            GoogleBackupRequest.GoogleBucketOptions.Builder googleBucketOptions = GoogleBackupRequest.GoogleBucketOptions.newBuilder().setBucketName(bucketOptions.getBucketName()).setFileKey(bucketOptions.getFileKey());
            if (!StringUtils.isBlank((CharSequence)bucketOptions.getAuthenticationFile())) {
                googleBucketOptions.setAuthenticationFile(bucketOptions.getAuthenticationFile());
            }
            GoogleBackupRequest gBackupRequest = GoogleBackupRequest.newBuilder().setBackupOptions(backupRequestBuilder).setGoogleBucketOptions(googleBucketOptions).build();
            String nodeAddress = client.getAddress();
            logger.info("Sending backup request to node {}", (Object)nodeAddress);
            this.getBackupOperationStatus().setAdditionalClusterNodeBackUpInfo(nodeAddress);
            if (client.requestGoogleBackup(gBackupRequest).getSuccess()) {
                logger.info("Cloud backup successfully performed.");
            }
        }
        finally {
            this.recoveryOperationInProgress.set(null);
        }
    }

    private void loggingMessagesForStartingS3BackupAndUpload(S3BucketOptions bucketOptions) {
        logger.info("Creating backup and uploading file: {}", (Object)bucketOptions.getFileKey());
        logger.info("To bucket: {}", (Object)bucketOptions.getBucketName());
        logger.info("On bucket host: {}", bucketOptions.getHostAsUri() != null ? String.valueOf(bucketOptions.getHostAsUri()) + "/" : "Amazon Web Services");
    }

    private void loggingMessagesForStartingAzureBackupAndUpload(AzureBlobOptions blobOptions) {
        logger.info("Creating backup and uploading file: {}", (Object)blobOptions.getFileKey());
        logger.info("To container: {}", (Object)blobOptions.getContainerName());
    }

    private void loggingMessagesForStartingGoogleBackupAndUpload(GoogleBucketOptions bucketOptions) {
        logger.info("Creating backup and uploading file: {}", (Object)bucketOptions.getFileKey());
        logger.info("To bucket: {}", (Object)bucketOptions.getBucketName());
    }

    private boolean isS3AccelerationEnabled(S3BucketOptions bucketOptions) {
        boolean enabled;
        try (S3Client s3 = S3Utils.s3ClientBuilder((S3BucketOptions)bucketOptions, (boolean)false);){
            enabled = S3Utils.checkBucketAcceleratedStatus((S3Client)s3, (String)bucketOptions.getBucketName());
            logger.info("Bucket accelerated status detected as: {}", (Object)enabled);
            if (enabled && bucketOptions.getHostAsUri() != null) {
                logger.info("The host override cannot be combined with S3 Accelerate, will not benefit from the bucket acceleration.");
                enabled = false;
            }
        }
        return enabled;
    }

    public RecoveryStatus getRecoveryOperationInProgress() {
        return this.recoveryOperationInProgress.get();
    }

    public BackupOperationStatus getBackupOperationStatus() {
        return this.backupOperationStatus.get();
    }

    @VisibleForTesting
    public void setBackupOperationStatus(BackupOperationStatus backupOperationStatus) {
        this.backupOperationStatus.set(backupOperationStatus);
    }

    private void setRecoveryOperationInfo(BackupOperationStatus.BackupOperationInProgress backupOperationInProgress, SnapshotOptions snapshotOptions) {
        Set<String> affectedRepositories = this.resolveAffectedRepositories(backupOperationInProgress, snapshotOptions);
        this.backupOperationStatus.set(new BackupOperationStatus(backupOperationInProgress, snapshotOptions, System.currentTimeMillis(), GraphDBHTTPContext.getAuthenticatedUser().getUsername(), affectedRepositories));
    }

    private void clearRecoveryOperationInfo() {
        this.backupOperationStatus.set(null);
    }

    public SnapshotMetadata getSnapshotMetadata() {
        return this.snapshotMetadata;
    }

    public void setSnapshotMetadata(SnapshotMetadata snapshotMetadata) {
        this.snapshotMetadata = snapshotMetadata;
    }

    public Set<String> resolveAffectedRepositories(BackupOperationStatus.BackupOperationInProgress backupOperationInProgress, SnapshotOptions snapshotOptions) {
        HashSet<String> result = new HashSet<String>();
        HashSet reposFromBackupFile = new HashSet();
        Set locationRepos = this.semanticDataManagement.getCurrentLocationOrThrow().getRepositoryIDs();
        SnapshotMetadata metadata = this.getSnapshotMetadata();
        if (metadata != null && metadata.getRepositories() != null) {
            reposFromBackupFile = new HashSet(metadata.getRepositories());
        }
        List snapShotRepos = snapshotOptions.getRepositories();
        if (snapshotOptions.isCleanDataDir()) {
            result.addAll(locationRepos);
        }
        if (snapShotRepos == null) {
            if (backupOperationInProgress == BackupOperationStatus.BackupOperationInProgress.CREATE_BACKUP_IN_PROGRESS || backupOperationInProgress == BackupOperationStatus.BackupOperationInProgress.CREATE_CLOUD_BACKUP_IN_PROGRESS) {
                result.addAll(locationRepos);
                return result;
            }
            if (backupOperationInProgress == BackupOperationStatus.BackupOperationInProgress.RESTORE_BACKUP_IN_PROGRESS || backupOperationInProgress == BackupOperationStatus.BackupOperationInProgress.RESTORE_CLOUD_BACKUP_IN_PROGRESS) {
                result.addAll(reposFromBackupFile);
                return result;
            }
        } else {
            result.addAll(snapShotRepos);
        }
        return result;
    }

    public static enum RecoveryStatus {
        CREATE_BACKUP_IN_PROGRESS("Can not %s while creating backup."),
        APPLY_BACKUP_IN_PROGRESS("Can not %s while applying backup.");

        final String errorMsg;

        private RecoveryStatus(String errorMsg) {
            this.errorMsg = errorMsg;
        }

        public String getErrorMsg() {
            return this.errorMsg;
        }
    }
}

