/*
 * Decompiled with CFR 0.152.
 */
package com.ontotext.forest.core;

import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import org.apache.commons.collections4.MapUtils;
import org.eclipse.rdf4j.query.BindingSet;
import org.eclipse.rdf4j.query.TupleQuery;
import org.eclipse.rdf4j.query.TupleQueryResult;
import org.eclipse.rdf4j.repository.RepositoryConnection;
import org.json.simple.JSONArray;
import org.json.simple.JSONObject;
import org.json.simple.parser.JSONParser;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SimilarityIndexesUtil {
    private static final Logger LOGGER = LoggerFactory.getLogger(SimilarityIndexesUtil.class);
    private static final String ELASTICSEARCH = "elasticsearch";
    private static final String OPENSEARCH = "opensearch";
    private static final Map<String, String> CONNECTOR_PREFIXES = Map.of("elasticsearch", "http://www.ontotext.com/connectors/elasticsearch#", "opensearch", "http://www.ontotext.com/connectors/opensearch#");
    private static final Set<String> VECTOR_DATATYPES = Set.of("vector", "native:vector");
    private static final Map<String, Set<String>> CONNECTOR_SPECIFIC_DATATYPES = Map.of("elasticsearch", Set.of("native:semantic_text"), "opensearch", Set.of("native:semantic"));

    public static Map<String, List<String>> getSimilarityIndexes(String repositoryId, RepositoryConnection connection) {
        LOGGER.debug("TTYG getSimilarityIndexes: ({})", (Object)repositoryId);
        JSONParser parser = new JSONParser();
        Map<String, List<String>> connectorVectors = SimilarityIndexesUtil.collectConnectorData(connection, parser, SimilarityIndexesUtil::processConnector);
        connectorVectors.putAll(SimilarityIndexesUtil.fetchSimilarityIndexes(connection));
        LOGGER.debug("Discovered total {} similarity search indexes.", (Object)connectorVectors.size());
        return connectorVectors;
    }

    public static Map<String, List<SimilarityIndex>> getSimilarityIndexesForMCP(String repositoryId, RepositoryConnection connection) {
        LOGGER.debug("TTYG getSimilarityIndexes: ({})", (Object)repositoryId);
        JSONParser parser = new JSONParser();
        Map<String, List<SimilarityIndex>> connectorVectors = SimilarityIndexesUtil.collectConnectorData(connection, parser, SimilarityIndexesUtil::processConnectorForMCP);
        connectorVectors.putAll(SimilarityIndexesUtil.fetchSimilarityIndexesForMCP(connection));
        LOGGER.debug("Discovered total {} similarity search indexes.", (Object)connectorVectors.size());
        return connectorVectors;
    }

    private static <T> Map<String, List<T>> collectConnectorData(RepositoryConnection connection, JSONParser parser, ConnectorProcessor<T> processor) {
        LinkedHashMap<String, List<T>> connectorVectors = new LinkedHashMap<String, List<T>>();
        for (Map.Entry<String, String> entry : CONNECTOR_PREFIXES.entrySet()) {
            connectorVectors.putAll(processor.process(connection, entry.getKey(), entry.getValue(), parser));
        }
        return connectorVectors;
    }

    private static Map<String, List<String>> processConnector(RepositoryConnection connection, String connectorType, String prefix, JSONParser parser) {
        return SimilarityIndexesUtil.processConnectorInternal(connection, connectorType, prefix, parser, (connectorName, vectorFields) -> Map.entry(connectorType + ":" + connectorName, vectorFields));
    }

    private static Map<String, List<SimilarityIndex>> processConnectorForMCP(RepositoryConnection connection, String connectorType, String prefix, JSONParser parser) {
        return SimilarityIndexesUtil.processConnectorInternal(connection, connectorType, prefix, parser, (connectorName, vectorFields) -> Map.entry(connectorType, List.of(new SimilarityVectorIndex(connectorName, vectorFields))));
    }

    private static <T> Map<String, List<T>> processConnectorInternal(RepositoryConnection connection, String connectorType, String prefix, JSONParser parser, ConnectorResultMapper<T> resultMapper) {
        LinkedHashMap<String, List<T>> connectorVectors = new LinkedHashMap<String, List<T>>();
        LOGGER.debug("Processing {} connectors...", (Object)connectorType);
        String listConnectorsQuery = "PREFIX prefix: <%s>\nSELECT ?uri ?name ?status WHERE {\n    ?uri prefix:listConnectors ?name ;\n         prefix:connectorStatus ?status .\n    FILTER regex(?status, \"BUILT\")\n}\n".formatted(prefix);
        TupleQuery listConnectors = connection.prepareTupleQuery(listConnectorsQuery);
        try (TupleQueryResult connectorResults = listConnectors.evaluate();){
            while (connectorResults.hasNext()) {
                BindingSet binding = (BindingSet)connectorResults.next();
                if (binding.getValue("name") == null) {
                    LOGGER.warn("Missing 'name' binding in connector result: {}", (Object)binding);
                    continue;
                }
                String connectorName = binding.getValue("name").stringValue();
                String instanceUri = SimilarityIndexesUtil.buildInstanceUri(prefix, connectorName);
                List<String> vectorFields = SimilarityIndexesUtil.fetchConnectorVectorFields(connection, connectorType, instanceUri, parser);
                if (!vectorFields.isEmpty()) {
                    Map.Entry<String, List<T>> entry = resultMapper.map(connectorName, vectorFields);
                    connectorVectors.merge(entry.getKey(), entry.getValue(), (oldList, newList) -> {
                        ArrayList merged = new ArrayList(oldList);
                        merged.addAll(newList);
                        return merged;
                    });
                    LOGGER.debug("Found {} vector fields for {} instance {}: {}", new Object[]{vectorFields.size(), connectorType, connectorName, vectorFields});
                    continue;
                }
                LOGGER.debug("No vector fields found for {} instance {}", (Object)connectorType, (Object)connectorName);
            }
        }
        catch (Exception e) {
            LOGGER.error("Error while processing {} connectors: {}", new Object[]{connectorType, e.getMessage(), e});
        }
        return connectorVectors;
    }

    private static Map<String, List<SimilarityIndex>> fetchSimilarityIndexesForMCP(RepositoryConnection connection) {
        String similarityQuery = "PREFIX : <http://www.ontotext.com/graphdb/similarity/>\nPREFIX similarity-index: <http://www.ontotext.com/graphdb/similarity/instance/>\nSELECT ?name WHERE {\n    ?index :status ?status ;\n           :name ?name.\n    FILTER regex(?status, \"BUILT|OUTDATED\")\n}";
        return SimilarityIndexesUtil.fetchSimilarityIndexesCommon(connection, similarityQuery);
    }

    private static Map<String, List<String>> fetchSimilarityIndexes(RepositoryConnection connection) {
        String similarityQuery = "PREFIX : <http://www.ontotext.com/graphdb/similarity/>\nPREFIX similarity-index: <http://www.ontotext.com/graphdb/similarity/instance/>\nSELECT ?index ?status WHERE {\n  ?index :status ?status .\n  FILTER regex(?status, \"BUILT|OUTDATED\")\n}\n";
        LinkedHashMap<String, List<String>> result = new LinkedHashMap<String, List<String>>();
        ArrayList<String> similarityIndexes = new ArrayList<String>();
        TupleQuery similarityTuple = connection.prepareTupleQuery(similarityQuery);
        try (TupleQueryResult similarityResults = similarityTuple.evaluate();){
            while (similarityResults.hasNext()) {
                BindingSet bs = (BindingSet)similarityResults.next();
                if (bs.getValue("index") == null) {
                    LOGGER.warn("Skipping row with missing 'index' binding: {}", (Object)bs);
                    continue;
                }
                String indexUri = bs.getValue("index").stringValue();
                String indexName = indexUri.substring(Math.max(indexUri.lastIndexOf(35), indexUri.lastIndexOf(47)) + 1);
                similarityIndexes.add(indexName);
            }
        }
        catch (Exception e) {
            LOGGER.error("Error fetching similarity indexes: {}", (Object)e.getMessage(), (Object)e);
        }
        if (!similarityIndexes.isEmpty()) {
            result.put("similarity", similarityIndexes);
            LOGGER.debug("Found {} similarity indexes: {}", (Object)similarityIndexes.size(), similarityIndexes);
        }
        return result;
    }

    private static Map<String, List<SimilarityIndex>> fetchSimilarityIndexesCommon(RepositoryConnection connection, String query) {
        LinkedHashMap<String, List<SimilarityIndex>> similarityMap = new LinkedHashMap<String, List<SimilarityIndex>>();
        TupleQuery similarityTuple = connection.prepareTupleQuery(query);
        try (TupleQueryResult results = similarityTuple.evaluate();){
            while (results.hasNext()) {
                BindingSet bs = (BindingSet)results.next();
                if (bs.getValue("name") == null) {
                    LOGGER.warn("Skipping similarity row with missing 'name' binding: {}", (Object)bs);
                    continue;
                }
                String indexName = bs.getValue("name").stringValue();
                similarityMap.computeIfAbsent("similarity", k -> new ArrayList()).add(new SimilarityIndex(indexName));
            }
        }
        catch (Exception e) {
            LOGGER.error("Error fetching similarity indexes for MCP: {}", (Object)e.getMessage(), (Object)e);
        }
        if (MapUtils.isNotEmpty(similarityMap)) {
            LOGGER.debug("Found {} similarity indexes: {}", (Object)similarityMap.size(), similarityMap.values());
        }
        return similarityMap;
    }

    private static String buildInstanceUri(String prefix, String connectorName) {
        String instanceUri = prefix;
        if (!instanceUri.endsWith("/instance#")) {
            instanceUri = instanceUri.replace("#", "/instance#");
        }
        return instanceUri + connectorName;
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private static List<String> fetchConnectorVectorFields(RepositoryConnection connection, String connectorType, String instanceUri, JSONParser parser) {
        ArrayList<String> vectorFields = new ArrayList<String>();
        String optionsQuery = "PREFIX prefix: <%s>\nSELECT ?options WHERE {\n  <%s> prefix:listOptionValues ?options\n}\n".formatted(CONNECTOR_PREFIXES.get(connectorType), instanceUri);
        TupleQuery optionsTuple = connection.prepareTupleQuery(optionsQuery);
        try (TupleQueryResult optionsResult = optionsTuple.evaluate();){
            JSONObject options;
            if (!optionsResult.hasNext()) {
                ArrayList<String> arrayList = vectorFields;
                return arrayList;
            }
            BindingSet bs = (BindingSet)optionsResult.next();
            if (bs.getValue("options") == null) {
                LOGGER.warn("No 'options' binding found for instance {}", (Object)instanceUri);
                ArrayList<String> arrayList = vectorFields;
                return arrayList;
            }
            String optionsJson = bs.getValue("options").stringValue();
            try {
                options = (JSONObject)parser.parse(optionsJson);
            }
            catch (Exception e) {
                LOGGER.warn("Failed to parse options for {} instance {}: {}", new Object[]{connectorType, instanceUri, e.getMessage()});
                ArrayList<String> arrayList = vectorFields;
                if (optionsResult == null) return arrayList;
                optionsResult.close();
                return arrayList;
            }
            JSONArray fields = (JSONArray)options.get((Object)"fields");
            if (fields == null) return vectorFields;
            SimilarityIndexesUtil.collectVectorFields(fields, vectorFields, connectorType);
            vectorFields.replaceAll(f -> {
                int dollar = f.indexOf(36);
                return dollar > 0 ? f.substring(0, dollar) : f;
            });
            vectorFields = new ArrayList<String>(new LinkedHashSet<String>(vectorFields));
            return vectorFields;
        }
        catch (Exception e) {
            LOGGER.error("Error fetching vector fields for {} instance {}: {}", new Object[]{connectorType, instanceUri, e.getMessage(), e});
        }
        return vectorFields;
    }

    private static void collectVectorFields(JSONArray fieldsArray, List<String> collector, String connectorType) {
        for (Object fieldObj : fieldsArray) {
            Object nested;
            if (!(fieldObj instanceof JSONObject)) continue;
            JSONObject fieldJson = (JSONObject)fieldObj;
            String datatype = (String)fieldJson.get((Object)"datatype");
            String fieldName = (String)fieldJson.get((Object)"fieldName");
            if (fieldName != null && datatype != null && (VECTOR_DATATYPES.contains(datatype) || CONNECTOR_SPECIFIC_DATATYPES.getOrDefault(connectorType, Set.of()).contains(datatype))) {
                collector.add(fieldName);
            }
            if (!((nested = fieldJson.get((Object)"objectFields")) instanceof JSONArray)) continue;
            JSONArray nestedArray = (JSONArray)nested;
            SimilarityIndexesUtil.collectVectorFields(nestedArray, collector, connectorType);
        }
    }

    @FunctionalInterface
    private static interface ConnectorProcessor<T> {
        public Map<String, List<T>> process(RepositoryConnection var1, String var2, String var3, JSONParser var4);
    }

    @FunctionalInterface
    private static interface ConnectorResultMapper<T> {
        public Map.Entry<String, List<T>> map(String var1, List<String> var2);
    }

    public static class SimilarityIndex {
        String similarityIndex;

        public SimilarityIndex(String similarityIndex) {
            this.similarityIndex = similarityIndex;
        }

        public String getSimilarityIndex() {
            return this.similarityIndex;
        }
    }

    public static class SimilarityVectorIndex
    extends SimilarityIndex {
        List<String> connectorFields;

        public SimilarityVectorIndex(String similarityIndex, List<String> connectorFields) {
            super(similarityIndex);
            this.connectorFields = new ArrayList<String>(connectorFields);
        }

        public List<String> getConnectorFields() {
            return this.connectorFields;
        }
    }
}

